<?php

namespace App\Http\Controllers\Admin;

use App\Models\City;
use App\Models\Sale;
use App\Models\Order;
use App\Models\Balance;
use App\Models\Product;
use App\Models\PurchaseItem;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\ResellerPaymentTransaction;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }


    public function getAnalysisData()
    {

        $stock = array();
        $stock['total_price'] = 0;
        $products = Product::where('stock', '>', 0)->get();
        $stock['total_quantity'] = $products->sum('stock');

        foreach ($products as $product) {
            $product_purchase_item = purchaseItem::where('product_id', $product->id)->get();
            $price = 0;

            foreach ($product_purchase_item as $item) {
                $price += $item->price * $item->stock;
            }
            if ($product_purchase_item->sum('stock') > 0) {
                $stock['total_price'] += ($price / $product_purchase_item->sum('stock')) * $product->stock;
            }
        }

        $admin_order = Order::adminOrderAnalysis();
        $topSellingProductToday = Order::topSellingProductToday();
        $analysis = Order::analysis();
        $due = Order::due();

        return response()->json([
            'stock' => $stock,
            'admin_order' => $admin_order,
            'top_selling_products_today' => $topSellingProductToday,
            'due' => $due,
            'analysis' => $analysis
        ]);
    }

    
    public function dashboard(Request $request)
    {

        // return $request->all();
        if (!empty($request->start_date) && (!empty($request->end_date))) {
            $order = Order::dateWiseOrderCount($request);
        } else {
            $order = Order::orderCount();
        }
        $currentYear = Carbon::now()->year;
        $query = Order::select(
            DB::raw('SUM(total) as order_amount'),
            DB::raw('SUM(discount) as order_discount'),
            DB::raw('MONTH(created_at) as month_number')
        );
        $startDate = $request->start_date ? Carbon::parse($request->start_date) : null;
        $endDate = $request->end_date ? Carbon::parse($request->end_date) : null;

        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()]);
        } else {
            $query->whereYear('created_at', $currentYear);
        }

        $monthlyOrdersRaw = $query->groupBy(DB::raw('MONTH(created_at)'))->get();

        // Calculate total orders for the year or date range
        $totalOrdersForYear = $monthlyOrdersRaw->sum('order_amount');

        // Prepare monthly data
        $monthlyOrders = collect(range(1, 12))->map(function ($monthNumber) use ($monthlyOrdersRaw, $totalOrdersForYear) {
            $monthData = $monthlyOrdersRaw->firstWhere('month_number', $monthNumber);

            $totalSales = $monthData ? $monthData->order_amount : 0;
            $percentage = $totalOrdersForYear > 0 ? ($totalSales / $totalOrdersForYear) * 100 : 0;

            return [
                'month_name' => Carbon::create()->month($monthNumber)->format('F'),
                'total_order_amount' => $totalSales,
                'total_order_discount' => $monthData ? $monthData->order_discount : 0,
                'percentage' => round($percentage, 2),
            ];
        });

        $total_orders = Order::count();
        $sourceQuery = Order::with('source')
            ->select('order_source')
            ->selectRaw('COUNT(*) as order_count')
            ->selectRaw('ROUND((COUNT(*) / ?) * 100, 2) as order_percentage', [$total_orders])
            ->groupBy('order_source');

        if ($startDate && $endDate) {
            $sourceQuery->whereBetween('created_at', [$startDate->startOfDay(), $endDate->endOfDay()]);
        }
        $orderCounts = $sourceQuery->get();
        return response()->json([
            'orders'                        => $order,
            'monthly_orders'                => $monthlyOrders,
            'order_source_percentages'      => $orderCounts,
        ]);
    }

    // public function dashboard(Request $request)
    // {
    //     if (!empty($request->start_date) && (!empty($request->end_date))) {
    //         $order = Order::dateWiseOrderCount($request);
    //     } else {
    //         $order = Order::orderCount();
    //     }
    //     $currentYear = Carbon::now()->year;
    //     $monthly_orders = Order::select(
    //         DB::raw('SUM(total) as order_amount'),
    //         DB::raw('SUM(discount) as order_discount'),
    //         DB::raw('MONTH(created_at) as month_number'),
    //     )
    //         ->whereYear('created_at', $currentYear)  // Filter by the current year
    //         ->groupBy(DB::raw('MONTH(created_at)'))
    //         ->get();

    //     // get the total order for the entire year
    //     $total_orders_for_year = $monthly_orders->sum('order_amount');

    //     // Initialize an array for all months
    //     $monthly_orders = collect(range(1, 12))->map(function ($monthNumber) use ($monthly_orders, $total_orders_for_year) {
    //         $monthData = $monthly_orders->firstWhere('month_number', $monthNumber);

    //         // calculate total orders and percentage for each month
    //         $totalSales = $monthData ? $monthData->order_amount : 0;
    //         $percentage = $total_orders_for_year > 0 ? ($totalSales / $total_orders_for_year) * 100 : 0;

    //         return [
    //             'month_name' => Carbon::create()->month($monthNumber)->format('F'),
    //             'total_order_amount' => $monthData ? $monthData->order_amount : 0,
    //             'total_order_discount' => $monthData ? $monthData->order_discount : 0,
    //             'percentage' => round($percentage, 2),
    //         ];
    //     });

    //     $total_orders = Order::count();
    //     $orderCounts = Order::with('source')
    //         ->select('order_source')
    //         ->selectRaw('COUNT(*) as order_count')
    //         ->selectRaw('ROUND((COUNT(*) / ?) * 100, 2) as order_percentage', [$total_orders])
    //         ->groupBy('order_source')
    //         ->get();
    //     return response()->json([
    //         'orders'                        => $order,
    //         'monthly_orders'                => $monthly_orders,
    //         'order_source_percentages'      => $orderCounts,
    //     ]);
    // }



    public function getAccounts()
    {
        $balance = Balance::with('today_credit_balance', 'total_credit_balance', 'today_debit_balance', 'total_debit_balance')->get();
        return response()->json([
            'balance' => $balance,
        ]);
    }
    
    public function districtWiseOrder(Request $request)
    {
    
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
    
            $total_orders = Order::whereDate('orders.created_at', '>=', $start_date)
                ->whereDate('orders.created_at', '<=', $end_date)->count();

            $city_order_counts = City::select('cities.id', 'cities.name')
                ->withCount(['order' => function ($query) use ($start_date, $end_date) {
                    $query->whereDate('orders.created_at', '>=', $start_date)
                          ->whereDate('orders.created_at', '<=', $end_date);
                }])
                ->orderByDesc('order_count')
                ->paginate($request->item ?? 10);
        } else {
            // Get total orders without date filtering
            $total_orders = Order::count();
    
            // Query cities with total order counts
            $city_order_counts = City::select('cities.id', 'cities.name')
                ->withCount('order')
                ->orderByDesc('order_count')
                ->paginate($request->item ?? 10);
        }
    
        // Transform city_order_counts to calculate position and percentage
        $city_order_counts->getCollection()->transform(function ($city, $index) use ($total_orders) {
            $city->position = $index + 1;
            $city->percentage = $total_orders > 0 ? round(($city->order_count / $total_orders) * 100, 2) : 0;
            return $city;
        });
    
        return response()->json([
            'status' => true,
            'cities' => $city_order_counts,
        ]);
    }
    
}
