<?php

namespace App\Http\Controllers\Admin;

use Throwable;
use App\Models\Product;
use App\Models\Description;
use App\Models\LandingPage;
use Illuminate\Support\Str;
use App\Services\FileUpload;
use Illuminate\Http\Request;
use App\Models\LandingPageReview;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;

class LandingPageController extends Controller
{
    public function pages()
    {
        $landing_pages = LandingPage::orderBy('id', 'desc')->get();
        return response()->json([
            'success' => true,
            'landing_pages' => $landing_pages,
        ]);
    }
    public function addPage(Request $request)
    {
        $data = $request->validate([
            'page_name'     => 'required|unique:landing_pages',
            'main_title'    => 'nullable',
            'hotline_title' => 'nullable',
            'hotline_phone' => 'nullable',
            'video_title'   => 'nullable',
            'video_url'     => 'nullable|url',
            'products'      => 'required|array',
            'page_slug'     => 'nullable',
            'image'         => 'nullable|image',
            'main_banner'   => 'required|image',
            'description_title' => 'nullable',
            'description_items_add' => 'required|array', 
            'images'        => 'required',
        ]);
        DB::beginTransaction();
        try {
            $data['page_slug'] = Str::slug($request->page_name);

            if (!empty($data['video_url'])) {
                $data['video_url'] = $this->convertToEmbedUrl($data['video_url']);
            }

            if ($request->hasFile('main_banner')) {
                $path = $request->file('main_banner')->store('images/page', 'public');
                $data['main_banner'] = $path;
                // return $path;
            }

            $page = LandingPage::create($data);

            foreach ($request->products as $item) {
                $product = Product::findOrFail($item['id']);
                $product->landing_page_id = $page->id;
                $product->save();
            }

            if ($data['images']) {              
                $files= $data['images'];
                if($files > 0) {
                    FileUpload::landingPageReview($files,$page->id);
                }
            }
            
            /***** added description ******/
            if (!empty($request->description_items_add) && is_array($request->description_items_add)) {
                foreach ($request->description_items_add as $descriptionDetails) {
                    Description::create([
                        'landing_page_id' => $page->id,
                        'title' => $descriptionDetails,
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Page added successfully',
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage()
            ]);
        }
    }



    function convertToEmbedUrl($url)
    {
        $pattern = '/(?:youtube\.com\/(?:[^\/\n\s]+\/\S+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]{11})/';

        if (preg_match($pattern, $url, $matches)) {
            return 'https://www.youtube.com/embed/' . $matches[1];
        }

        return $url; // Return original URL if it doesn't match YouTube patterns
    }

    public function update(Request $request, $id)
    {
        $data = $request->validate([
            'page_name'             => 'required',
            'page_slug'             => 'nullable|unique:landing_pages,page_slug,' . $id,
            'main_title'            => 'nullable',
            'hotline_title'         => 'nullable',
            'hotline_phone'         => 'nullable',
            'video_title'           => 'nullable',
            'video_url'             => 'nullable|url',
            'products'              => 'nullable|array',
            'image'                 => 'nullable|image',
            'main_banner'           => 'nullable',
            'description_title'     => 'nullable',
            'description_details'   => 'nullable',
            'images'                => 'nullable',
        ]);



        DB::beginTransaction();
        try {
            // Find the landing page
            $page = LandingPage::findOrFail($id);

            // Convert video URL to embed format if provided
            if (!empty($data['video_url'])) {
                $data['video_url'] = $this->convertToEmbedUrl($data['video_url']);
            }

            // Handle main banner upload
            if ($request->hasFile('main_banner')) {
                //delete existing banner
                $bannerExists    =   Storage::disk('public')->exists($page->main_banner);
                if($bannerExists == true) {
                    Storage::disk('public')->delete($page->main_banner);
                }
              
                $path = $request->file('main_banner')->store('images/page', 'public');
                $data['main_banner'] = $path;
            }

            /***** delete previous record *****/
            $description = Description::where('landing_page_id', $page->id)->get();
            foreach($description as $item){
                $item->delete();
            }


            // update description items
            foreach ($request->description_details as $item) {
                $description = new Description();
                $description->landing_page_id = $page->id;
                $description->title     = $item;
                $description->save();
            }
            
            if ($data['images']) {              
                $files= $data['images'];
                if($files > 0) {
                    FileUpload::landingPageReview($files,$page->id);
                }
            }

            if ($request->products) {
                foreach ($request->products as $item) {
                    $product = Product::findOrFail($item['id']);
                    $product->landing_page_id = $page->id;
                    $product->save();
                }
            }
            $page->update($data);

            
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Information updated successfully',
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage()
            ]);
        }

       


    }


    public function edit($id)
    {
        $page = LandingPage::with(['product:id,thumbnail_img,name,product_code,sale_price,price,stock,is_auto_cart,landing_page_id', 'landingPageReview', 'description'])->where('id', $id)->first();
        $reviews = LandingPageReview::where('landing_page_id',$page->id)->get();
        return response()->json([
            'status' => true,
            'page' => $page,
            'reviews' => $reviews,
        ]);
    }

    public function deleteImage($id)
    {
        LandingPageReview::findOrFail($id)->delete();
        return response()->json([
            'success' => true,
            'message' => ' image  deleted successfully !'
        ]);
    }


    public function status($id)
    {
        $page = LandingPage::findOrFail($id);
        if ($page->status == 1) {
            $page->status = 0;
            $page->save();
        } else {
            $page->status = 1;
            $page->save();
        }

        return response()->json([
            'success' => true,
            'message' => 'Updated successfully',
        ]);
    }



    public function activePage()
    {
        $landing_pages = LandingPage::where('status', 1)->orderBy('id', 'desc')->get();
        return response()->json([
            'success' => true,
            'pages' => $landing_pages,
        ]);
    }


    public function removeProduct($id)
    {
        $product = Product::findOrFail($id);
        if ($product) {
            $product->landing_page_id = null;
            $product->save();

            return response()->json([
                'status' => true,
                'message' => 'Product removed successfully',
            ]);
        } else {
            return response()->json([
                'status' => false,
            ]);
        }
    }

    public function updateProductAutoCartStatus(Request $request)
    {
        if ($request->product_id == null) {
            return response()->json([
                'status' => false,
                'message' => 'Please select any product',
            ]);
        }

        $product = Product::findOrFail($request->product_id);

        if ($request->checked == 1) {
            $product->is_auto_cart = 1;
            $message = 'Product added as Auto Cart';
        } elseif ($request->checked == 0) {
            $product->is_auto_cart = 0;
            $message = 'Product removed from Auto Cart ';
        }

        $product->save();

        return response()->json([
            'status' => true,
            'message' => $message,
        ]);
    }


    public function landingPageProduct($id){
        $products  = Product::where('landing_page_id', $id)->get();
        return response()->json([
            'status' => true,
            'products' => $products,
        ]);
    }
}
