<?php

namespace App\Http\Controllers\Admin;

use Exception;
use Throwable;
use Carbon\Carbon;
use App\Models\City;
use App\Models\Sale;
use App\Models\Admin;
use App\Models\Order;
use App\Models\Credit;
use App\Models\Balance;
use App\Models\Comment;
use App\Models\Courier;
use App\Models\Product;
use App\Models\SubCity;
use App\Models\Variant;
use App\Models\Customer;
use App\Models\Merchant;
use App\Models\Reseller;
use App\Models\OrderItem;
use App\Models\SubCategory;
use App\Filters\OrderFilter;
use App\Models\ComboProduct;
use App\Models\OrderComment;
use App\Services\CourierApi;
use App\Services\LogTracker;
use App\Services\SmsService;
use Illuminate\Http\Request;
use App\Exports\OrdersExport;
use App\Models\GeneralSetting;
use App\Models\ProductVariant;
use App\Services\HelperService;
use App\Models\CustomerCashbook;
use App\Models\OrderItemVariant;
use App\Models\ResellerCashbook;
use App\Services\AccountService;
use App\Models\SiteConfiguration;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use App\Models\ResellerPaymentMethod;
use Illuminate\Support\Facades\Response;
use App\Models\ResellerPaymentTransaction;
use App\Models\CourierApi as ModelsCourierApi;
use App\Http\Requests\Order\Admin\StoreRequest;
use App\Http\Requests\Order\Admin\UpdateRequest;
use App\Models\ResellerPaymentTransactionDetail;
use App\Http\Controllers\Reseller\HomeController;
use App\Models\ResellerCashbookPaymentTransactionDetail;
use App\Services\CourierOrderInfoService;

class OrderController extends Controller
{

    public function __construct(Request $request)
    {
        $this->middleware('admin');
    }




    public function index(Request $request)
    {
        $query  = OrderFilter::adminOrderFilter($request);
        $orders = $query->orderBy('id', 'desc')->paginate($request->item ?? 50);

        return response()->json([
            'status'             => true,
            'orders'             => $orders,
            'order_count'        => Order::orderStatusCount($request),
            'comments'           => Comment::all(),
            'couriers'           => Courier::where('status', 1)->get(),
        ]);
    }


    // public function orderStatistic(Request $request)
    // {
    //     return response()->json([
    //         'status' => 'OK',
    //         'order_count' => Order::orderStatusCount($request),
    //     ]);
    // }

    public function store(Request $request)
    {
        // return $request->all();
        $data = $request->validate([
            'customer_phone' => 'required|digits:11 ',
            'name' => 'required ',
            'address' => 'required ',
            'balance_id' => 'nullable',
            'city_id' => 'nullable',
            'courier_id' => 'nullable|integer',
            'shipping_cost' => 'required',
            'sub_city_id' => 'nullable',
            'order_type' => 'required',
            'total' => 'required|integer',
            // 'discount' => 'nullable|integer',
            // 'paid' => 'nullable|integer',
            'status' => 'required|integer',
            'note' => 'nullable',
            'redx_delivery_area_id' => 'nullable',
            'order_source' => 'nullable',
        ]);

        DB::beginTransaction();
        try {
            /*** first find the customer ***/
            $customer = Customer::where('phone', $data['customer_phone'])->first();
            $setting = SiteConfiguration::first();
            // !empty($customer) ? $customer->update($data) : $customer = Customer::storeCustomer($data);
            if (!empty($customer)) {
                $customer->update($data);
                $customer->total_order = $customer->total_order + 1;
                $customer->save();
            } else {
                $customer = new Customer();
                $customer->name = $data['name'];
                $customer->phone = $data['customer_phone'];
                $customer->address = $data['address'];
                $customer->customer_type = $data['order_type'] ?? 1;
                $customer->total_order = 1;
                $customer->password = Hash::make($data['customer_phone']);
                $customer->save();
                $first_name = explode(' ', trim(strtolower($customer->name)))[0];
                $username = $first_name;
                $counter = 1;
                while (Customer::where('username', $username)->exists()) {
                    $username = $first_name . $counter;
                    $counter++;
                }
                $customer->username = $username;
                $customer->city_id = $data['city_id'];
                $customer->sub_city_id = $data['sub_city_id'];
                $customer->save();
            }
            //order data
            $data['customer_id'] = $customer->id;
            $data['invoice_no'] = 123;
            $data['create_admin_id'] = session()->get('admin')['id'];
            //store who is approved
            if ($data['status'] == 3) {
                $data['approved_admin_id'] = session()->get('admin')['id'];
                $data['approved_date'] = Carbon::now();
            }
            //store who is pending
            if ($data['status'] == 2) {
                $data['pending_admin_id'] = session()->get('admin')['id'];
                $data['pending_date'] = Carbon::now();
            }
            //default courier
            $courier = Courier::where('is_default', 1)->select('id')->first();
            if (empty($data['courier_id']) && !empty($courier)) {
                $data['courier_id'] = $courier->id;
            }
            $order = Order::query()->create($data);
            //if order save then save the order details
            foreach ($request->products as $product) {
                $details             = new OrderItem();
                $details->order_id   = $order->id;
                $details->product_id = $product['id'];
                $details->price      = $product['price'];
                $details->quantity   = $product['quantity'];
                $details->total      = $product['quantity'] * $product['price'];

                $details->save();


                if (is_array($product['variants'])) {
                    // Iterate through variants if they exist
                    foreach ($product['variants'] as $attributeName => $variantName) {
                        OrderItemVariant::create([
                            'order_item_id' => $details->id,
                            'variant_atribute' => $attributeName ?? null,
                            'variant_name' => $variantName ?? null,
                        ]);
                    }
                }
                /*** checking pre stock order ***/
                if ($setting->is_pre_stock == 1) {
                    $products = Product::findOrFail($product['id']);
                    if ($setting->variant_wise_stock != 1) {
                        $products->stock = $products->stock - $product['quantity'];
                        $products->save();
                    } else {
                        if (is_array($product['variants'])) {
                            foreach ($product['variants'] as $attributeName => $variantName) {
                                $variant  = Variant::where('name', $variantName)->select('id', 'name')->first();
                                if ($variant) {
                                    $productVariant = ProductVariant::where('product_id', $product['id'])
                                        ->where('variant_id', $variant->id)
                                        ->first();
                                    if ($productVariant) {
                                        $productVariant->stock = $productVariant->stock - $product['quantity'];
                                        $productVariant->save();
                                    }
                                }
                            }
                        }
                    }
                    $details->purchase_price = $products->purchase_price;
                    $details->save();
                }
            }
            $order->discount = $request->discount > 0 ? $request->discount : 0;
            $order->paid = $request->paid > 0 ? $request->paid : 0;

            /**** making 6 digit unique invoice no ****/
            if (strlen($order->id) > 3) {
                $order->invoice_no = rand(1, 9) . $order->id . rand(1, 9);
            } else if (strlen($order->id) > 2) {
                $order->invoice_no = rand(11, 99) . $order->id . rand(1, 9);
            } else if (strlen($order->id) > 1) {
                $order->invoice_no = rand(11, 99) . $order->id . rand(11, 99);
            } else {
                $order->invoice_no = rand(111, 999) . $order->id . rand(11, 99);
            }

            $courier_report = CourierOrderInfoService::getCustomerOrderStats($request->customer_phone);

            $order->return_qty = $courier_report['total_return'];
            $order->delivered_qty = $courier_report['total_delivery'];


            $order->save();
            //create a credit
            if ($order->paid > 0) {
                $comment = "Order Paid Amount '$order->paid' Paid by '$order->balance_id'. Order Invoice Number '$order->invoice_no";
                AccountService::creditStore('Order Paid Amount', $order->paid, $order->balance_id, $order->id, null, $comment);
            }
            //create a barcode for order
            HelperService::orderBarcodeStore($order->id, $order->invoice_no);
            //send message
            (new SmsService())->SendMessageCustomer($order->customer_phone, $customer->name, $order->invoice_no);

            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Order created successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }





    public static function stockRemover($products)
    {
        foreach ($products as $product) {
            $product_stock = Product::where('id', $product['id'])->first();
            $product_stock->stock = $product_stock->stock - $product['quantity'];
            $product_stock->save();
        }
    }



    public function update(Request $request, $id)
    {
        // return $request->all();
        $data = $request->validate([
            'customer_phone' => 'required|digits:11 ',
            'name' => 'required ',
            'address' => 'required ',
            'city_id' => 'nullable',
            'courier_id' => 'nullable|integer',
            'shipping_cost' => 'required',
            'sub_city_id' => 'nullable',
            'order_type' => 'required',
            // 'discount' => 'nullable|integer',
            // 'paid' => 'nullable|integer',
            'total' => 'required|integer',
            'balance_id' => 'nullable',
            'note' => 'nullable',
            'status' => 'nullable',
            'redx_delivery_area_id' => 'nullable',
            'order_source' => 'nullable',
        ]);

        $order = Order::findOrFail($id);
        DB::beginTransaction();
        try {

            //first find find the customer
            $customer = Customer::where('phone', $data['customer_phone'])->first();
            $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();
            !empty($customer) ? $customer->update($data) : Customer::storeCustomer($data);
            $customer = Customer::where('phone', $data['customer_phone'])->first();

            if ($order->status == 4 || $order->status == 5) {
                return \response()->json('Order can not edit in shipment and delivered');
            }
            $data['discount'] = $request->discount > 0 ? $request->discount : 0;
            $data['paid'] = $request->paid > 0 ? $request->paid : 0;

            //update credit
            if ($order->order_type != 4 &&   $data['paid'] > 0) {
                $credit = Credit::where('order_id', $order->id)->first();
                if ($credit) {
                    $credit->amount = $data['paid'];
                    $credit->purpose = "order advance paid amount updated ";
                    $credit->comment = "order advance paid amount updated previous amount " . $credit->amount . ' updated amount ' . $data['paid'];
                    $credit->save();
                } else {
                    $comment = "order advance paid amount " . $data['paid'] . " the order invoice number is " . $order->invoice_no;
                    AccountService::creditStore('order advance payment', $data['paid'], $request->balance_id, $order->id, null, $comment);
                }
            }


            //expanse from  reseller wallet if the order paid amount  is greater than previous paid
            if ($order->order_type == 4 && ($data['paid'] != $order->paid)) {
                if ($data['paid'] != $order->paid) {
                    $reseller_cashbook = ResellerCashbook::where('order_id', $order->id)->where('reseller_id', $order->reseller_id)->first();
                    if ($reseller_cashbook) {
                        $reseller_cashbook->delete();
                    }
                    HelperService::resellerCashbookStore($order->id, $order->reseller_id, $data['paid'], 0, $data['paid'] . ' paid amount at the time of order updated, previous record deleted.');
                }
            }

            //delete packaging cost
            if (!empty($order->reseller_id) && $order->order_type == 4 && $data['status'] != 8) {
                $reseller_cashbook = ResellerCashbook::where('order_id', $order->id)->where('amount', 10)->first();
                if (!empty($reseller_cashbook)) {
                    $reseller_cashbook->delete();
                }
                $order->packaging_charge = 0;
            }
            if (empty($order->pending_admin_id)  && $data['status'] == 2) {
                $order->pending_admin_id = session()->get('admin')['id'];
                $order->pending_date = Carbon::now();
            }
            if (empty($order->approved_admin_id)  && $data['status'] == 3) {
                $order->approved_admin_id = session()->get('admin')['id'];
                $order->approved_date = Carbon::now();
            }

            $data['customer_id'] = $customer->id;
            $order->update($data);

            //delete previous items and stock in
            if ($setting->is_pre_stock == 1) {
                $orderItems     = OrderItem::where('order_id', $order->id)->get();
                $productIds     = $orderItems->pluck('product_id')->toArray();
                $orderItemIds   = $orderItems->pluck('id')->toArray();

                $products = Product::whereIn('id', $productIds)->get()->keyBy('id');

                if ($setting->variant_wise_stock != 1) {
                    foreach ($orderItems as $item) {
                        if (isset($products[$item->product_id])) {
                            $products[$item->product_id]->increment('stock', $item->quantity);
                        }
                    }
                } else {
                    foreach ($request->products as $product) {
                        if ($product['status'] == 1) {
                            if (is_array($product['variants'])) {
                                foreach ($product['variants'] as $attributeName => $variantName) {
                                    $variant  = Variant::where('name', $variantName)->select('id', 'name')->first();
                                    if ($variant) {
                                        $productVariant = ProductVariant::where('product_id', $product['product_id'])
                                            ->where('variant_id', $variant->id)
                                            ->first();
                                        if ($productVariant) {
                                            $productVariant->stock = $productVariant->stock + $product['quantity'];
                                            $productVariant->save();
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                OrderItemVariant::whereIn('order_item_id', $orderItemIds)->delete();
                OrderItem::where('order_id', $order->id)->delete();
            } else {
                $orderItems = OrderItem::where('order_id', $order->id)->get();
                $orderItemIds = $orderItems->pluck('id')->toArray();
                OrderItemVariant::whereIn('order_item_id', $orderItemIds)->delete();
                OrderItem::where('order_id', $order->id)->delete();
            }

            foreach ($request->products as $product) {
                $details                 = new OrderItem();
                $details->order_id       = $order->id;
                $details->product_id     = $product['product_id'];
                $details->price          = $product['price'];
                $details->quantity       = $product['quantity'];
                $details->total          = $product['quantity'] * $product['price'];
                // $details->purchase_price = $product['purchase_price'];
                $details->save();

                /*** checking pre stock order ***/
                if ($setting->is_pre_stock == 1) {
                    $products = Product::findOrFail($product['product_id']);
                    if ($setting->variant_wise_stock != 1) {
                        $products->stock = $products->stock - $product['quantity'];
                        $products->save();
                    } else {
                        if (is_array($product['variants'])) {
                            foreach ($product['variants'] as $attributeName => $variantName) {
                                $variant  = Variant::where('name', $variantName)->select('id', 'name')->first();
                                if ($variant) {
                                    $productVariant = ProductVariant::where('product_id', $product['product_id'])
                                        ->where('variant_id', $variant->id)
                                        ->first();
                                    if ($productVariant) {
                                        $productVariant->stock = $productVariant->stock - $product['quantity'];
                                        $productVariant->save();
                                    }
                                }
                            }
                        }
                    }
                    $details->purchase_price = $products->purchase_price;

                    $details->save();
                }
                if (is_array($product['variants'])) {
                    // Iterate through variants if they exist
                    foreach ($product['variants'] as $attributeName => $variantName) {
                        OrderItemVariant::create([
                            'order_item_id' => $details->id,
                            'variant_atribute' => $attributeName ?? null,
                            'variant_name' => $variantName ?? null,
                        ]);
                    }
                }
            }

            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Order updated successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }





    public function approved($id)
    {
        // return $response = (new TrackingOrder())->tracking('438723B61');
        $order = Order::findOrFail($id);
        try {
            $order->status = 3;
            if (empty($order->approved_admin_id)) {
                $order->approved_admin_id = session()->get('admin')['id'];
                $order->approved_date = Carbon::now();
            }
            $order->save();
            return response()->json([
                'status' => 'SUCCESS',
                'message' => 'Order  approved  successfully',
            ]);
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }






    public function packaging($id)
    {
        $order = Order::findOrFail($id);
        try {
            $order->status = 8;
            $order->packaging_admin_id = session()->get('admin')['id'];
            $order->packaging_date = Carbon::now();
            $order->save();
            return response()->json([
                'status' => 'SUCCESS',
                'message' => 'Order  packaging  successfully done',
            ]);
        } catch (Exception $e) {
            return $e->getMessage();
        }
    }



    public function pending($id)
    {
        $order = Order::findOrFail($id);
        $order->status = 2;
        $order->pending_admin_id = session()->get('admin')['id'];
        $order->pending_date = Carbon::now();
        $order->save();
        return response()->json([
            'status' => 'SUCCESS',
            'message' => 'Order  pending  successfully',
        ]);
    }









    public function delivered(Request $request)
    {

        $data = $request->validate([
            'order_id' => 'required',
        ]);
        $order = Order::findOrFail($data['order_id']);

        if ($order->status != 4 && $order->status != 10) {
            return response()->json("order not shipment");
        }
        DB::beginTransaction();
        try {

            if (!empty($order->reseller_id) && $order->order_type == 4) {
                $reseller = Reseller::findOrFail($order->reseller_id);
                $order_items = OrderItem::where('order_id', $order->id)->get();
                $cod_amount =  (intval($order->total) + intval($order->shipping_cost)) - (intval($order->discount) + intval($order->paid));
                // $order->profit = $cod_amount;
                $note = 'successful order  delivered commission received ' . $order->profit . ' BDT order discount ' . $order->discount . ' BDT the order invoice number is ' . $order->invoice_no;
                HelperService::resellerCashbookStore($order->id, $reseller->id, $order->profit, 1, $note);
            }

            // Handle affiliate commission
            if (!empty($order->affiliate_customer_id)) {
                $affiliate_amount = 0;
                $order_items = OrderItem::where('order_id', $order->id)->get();

                foreach ($order_items as $item) {
                    $product = Product::findOrFail($item->product_id);
                    if ($product && $product->affiliate_profit > 0) {
                        $affiliate_amount += $product->affiliate_profit * $item->quantity;
                    }
                }

                if ($affiliate_amount > 0) {
                    $affiliate_customer = Customer::findOrFail($order->affiliate_customer_id);
                    $affiliate_customer->amount = $affiliate_customer->amount + $affiliate_amount;
                    $affiliate_customer->save();

                    CustomerCashbook::create([
                        'order_id' => $order->id,
                        'amount' => $affiliate_amount,
                        'customer_id' => $affiliate_customer->id,
                        'type' => 'affiliate',
                        'note' => "Profit added for delivered order invoice no " . $order->invoice_no . ' amount is ' . $affiliate_amount . ' TK',
                    ]);
                }
            }

            $order_items = OrderItem::where('order_id', $order->id)->get();

            foreach ($order_items as $item) {
                //for exchange order items. re-stocking exchange items
                if ($item->status == 3) {
                    $exchange_item = Product::where('id', $item->product_id)->firstOrFail();
                    $exchange_item->stock = $exchange_item->stock + $item->qty;
                    $exchange_item->save();
                }
            }
            foreach ($order_items as $item) {
                $product = Product::findOrFail($item->product_id);
                if (!empty($product->merchant_id)) {
                    $sub_category = SubCategory::find($product->sub_category_id);
                    $sub_category->merchant_commission;
                    $merchant = Merchant::findOrFail($product->merchant_id);
                    if ($sub_category && $sub_category->merchant_commission > 0) {
                        $commission_price = (($product->price * $item->quantity) * $sub_category->merchant_commission) / 100;
                        $commission_amount = ($product->price * $item->quantity) - $commission_price;
                        HelperService::merchantCashbookStore($order->id, $merchant->id, $commission_amount, 1, 'commission paid the order invoice number is ' . $order->invoice_no);
                    }
                }
            }
            $order->status = 5;
            $order->delivered_admin_id = session()->get('admin')['id'];
            $order->delivery_date = Carbon::now();
            $order->paid = ($order->total  + $order->shipping_cost) - $order->discount;
            $order->is_profit_paid = 0;
            $order->save();
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Order delivered successfully'
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }



    public function shipment($id)
    {
        $order = Order::findOrFail($id);
        DB::beginTransaction();
        try {
            /*** checking pre stock order ***/
            $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();

            if ($setting && $setting->is_pre_stock == 0) {
                //check product stock before order approved
                $order_items = OrderItem::where('order_id', $order->id)->get();


                foreach ($order_items as $order_item) {

                    $product = Product::where('id', $order_item->product_id)->firstOrFail();


                    if ($product->stock < $order_item->quantity) {
                        return response()->json([
                            'status' => false,
                            'message' => 'stock is less than request quantity, For product ' . $product->name . ' the code ' . $product->product_code . ' current stock is ' . $product->stock .
                                ' and request quantity is ' . $order_item->quantity,
                        ]);
                    }

                    //combo campaign products
                    if ($product->is_combo == 1) {

                        $combo_products_id  = ComboProduct::where('combo_product_id', $product->id)->select('general_product_id')->pluck('general_product_id');
                        $combo_products     = Product::whereIn('id', $combo_products_id)->select('id', 'stock', 'product_code')->get();

                        //save purchase price for profit calculation
                        $combo_products_purchase_price  = Product::whereIn('id', $combo_products_id)->sum('purchase_price');
                        $order_item->purchase_price     = $combo_products_purchase_price;

                        foreach ($combo_products as $item) {

                            if ($setting->variant_wise_stock != 1) {
                                $this->productStockDecrease($item, $order_item);
                            } else {
                                $this->comboVariantStockDecrease($item, $order_item);
                            }

                            // $item->stock = $item->stock - $order_item->quantity;
                            // $item->save();
                            //for exchange order items. re-stocking exchange items
                            if ($order_item->status == 3) {
                                $product = Product::findOrFail($order_item->product_id);
                                $this->productStockIncrease($product, $order_item);
                            }
                        }
                    } else {
                        //for exchange order items. re-stocking exchange items
                        if ($order_item->status == 3) {
                            $product = Product::findOrFail($order_item->product_id);
                            $this->productStockIncrease($product, $order_item);
                        } else {
                            if ($setting->variant_wise_stock != 1) {
                                $this->productStockDecrease($product, $order_item);
                            } else {
                                $this->variantStockDecrease($order_item);
                            }

                            $order_item->purchase_price = $product->purchase_price;
                        }
                    }
                    $order_item->save();
                }
            }

            $order->status = 4;
            $order->shipment_admin_id = session()->get('admin')['id'];
            $order->shipment_date = Carbon::now();
            $order->save();

            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Order shipped successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function return($id)
    {
        $order = Order::findOrFail($id);
        DB::beginTransaction();
        try {
            $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();

            $order_items = OrderItem::where('order_id', $order->id)->get();
            foreach ($order_items as $order_item) {
                if ($order_item->status == 1) {
                    $product = Product::where('id', $order_item->product_id)->firstOrFail();
                    //if combo campaign products
                    if ($product->is_combo == 1) {
                        $combo_products_id  = ComboProduct::where('combo_product_id', $product->id)->select('general_product_id')->pluck('general_product_id');
                        $combo_products     = Product::whereIn('id', $combo_products_id)->select('id', 'stock', 'product_code')->get();
                        foreach ($combo_products as $item) {
                            if ($setting->variant_wise_stock != 1) {
                                $this->productStockIncrease($item, $order_item);
                            } else {
                                $this->comboVariantStockIncrease($item, $order_item);
                            }

                            // $item->stock = $item->stock + $order_item->quantity;
                            // $item->save();
                        }
                    } else {
                        if ($setting->variant_wise_stock != 1) {
                            $this->productStockIncrease($product, $order_item);
                        } else {
                            $this->variantStockIncrease($order_item);
                        }
                    }
                }
            }

            $order->status = 7;
            $order->return_admin_id = session()->get('admin')['id'];
            $order->is_profit_paid = 0;
            $order->return_date = Carbon::now();
            $order->save();

            if (!empty($order->reseller_id) && $order->order_type == 4) {
                //expense from reseller wallet
                HelperService::resellerCashbookStore($order->id, $order->reseller_id, $order->regular_shipping_cost, 0,  'order shipping cost BDT ' . $order->regular_shipping_cost);
            }

            //if order is reseller order. and order has paid amount then delete previous reseller cashbook
            if ($order->paid > 0 && !empty($order->reseller_id) && $order->order_type == 4) {
                //checking if order is paid by admin
                $credit_check = Credit::where('order_id', $order->id)->first();
                if (!$credit_check) {

                    $order = Order::findOrFail($order->id);
                    $order->paid = 0;
                    $order->advance_paid = 0;
                    $order->save();
                    $reseller_cashbook = ResellerCashbook::where('order_id', $order->id)->where('reseller_id', $order->reseller_id)->firstOrFail();
                    $reseller_cashbook->delete();
                }
            }

            DB::commit();
            return response()->json([
                'status' => 'SUCCESS',
                'message' => 'Order  returned successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }


    public function WFP($id)
    {

        $order = Order::findOrFail($id);
        $order->wfp_admin_id = session()->get('admin')['id'];
        $order->wfp_date = Carbon::now();
        $order->status = 9;
        $order->save();
        return response()->json([
            'status' => 'SUCCESS',
            'message' => 'Added product out of stock section.'
        ]);
    }


    public function cancel(Request $request, $id)
    {
        // return $request->all();

        $order = Order::findOrFail($id);
        $setting = SiteConfiguration::first();
        DB::beginTransaction();
        try {
            // if ($order->status != 1 && $order->status != 2 && $order->status != 3) {
            $order_items = OrderItem::where('order_id', $order->id)->get();

            if ($setting->is_pre_stock == 1) {
                foreach ($order_items as $order_item) {
                    $product = Product::where('id', $order_item->product_id)->first();

                    if ($product->is_combo == 1) {
                        $combo_products_id  = ComboProduct::where('combo_product_id', $product->id)->select('general_product_id')->pluck('general_product_id');

                        $combo_products     = Product::whereIn('id', $combo_products_id)->select('id', 'stock', 'product_code')->get();

                        foreach ($combo_products as $item) {

                            if ($setting->variant_wise_stock != 1) {
                                $this->productStockIncrease($item, $order_item);
                            } else {
                                $this->comboVariantStockIncrease($item, $order_item);
                            }
                        }
                    } else {

                        if ($setting->variant_wise_stock != 1) {
                            $this->productStockIncrease($product, $order_item);
                        } else {
                            $this->variantStockIncrease($order_item);
                        }
                    }
                }
            }
            // }

            $order->status = 6;
            $order->cancel_admin_id = session()->get('admin')['id'];
            $order->cancel_date = Carbon::now();

            //if order is reseller order. and order has paid amount then restore reseller cashbook
            if (!empty($order->reseller_id) && $order->order_type == 4) {
                $reseller_cashbook = ResellerCashbook::where('order_id', $order->id)->get();
                if (!empty($reseller_cashbook)) {
                    foreach ($reseller_cashbook as $item) {
                        $item->delete();
                    }
                }
                $order->advance_paid = 0;
                $order->paid = 0;
            }

            $order->save();

            $customer = Customer::where('phone', $order->customer_phone)->first();
            if ($request->status == 4) {
                $customer->status = $request->status;
            }
            // $customer->is_blocked = $request->is_block;
            $customer->save();




            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Order canceled successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            LogTracker::failLog($e, session()->get('admin')['id']);
            return response()->json([
                'status' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function ipBlockForCustomer(Request $request, $id)
    {
        // return $id;
        $order = Order::findOrFail($id);
        $customer = Customer::where('phone', $order->customer_phone)->first();
        $customer->is_blocked = $request->is_block;
        $customer->save();

        if ($request->is_block == 1) {
            return response()->json([
                'status' => true,
                'message' => 'IP unblocked successfully',
            ]);
        } else {
            return response()->json([
                'status' => true,
                'message' => 'IP blocked successfully',
            ]);
        }
    }



    protected function variantStockIncrease($orderItem)
    {

        $orderItemVariants = OrderItemVariant::where('order_item_id', $orderItem->id)->get();

        foreach ($orderItemVariants as $orderItemVariant) {
            $variant  = Variant::where('name', $orderItemVariant->variant_name)->select('id', 'name')->first();
            if ($variant) {
                $productVariant = ProductVariant::where('product_id', $orderItem->product_id)
                    ->where('variant_id', $variant->id)
                    ->first();
                if ($productVariant) {
                    $productVariant->stock = $productVariant->stock + $orderItem->quantity;
                    $productVariant->save();
                }
            }
        }
    }


    protected function variantStockDecrease($orderItem)
    {
        $orderItemVariants = OrderItemVariant::where('order_item_id', $orderItem->id)->get();

        foreach ($orderItemVariants as $orderItemVariant) {
            $variant  = Variant::where('name', $orderItemVariant->variant_name)->select('id', 'name')->first();
            if ($variant) {
                $productVariant = ProductVariant::where('product_id', $orderItem->product_id)
                    ->where('variant_id', $variant->id)
                    ->first();
                if ($productVariant) {
                    $productVariant->stock = $productVariant->stock - $orderItem->quantity;
                    $productVariant->save();
                }
            }
        }
    }

    protected function comboVariantStockDecrease($product, $orderItem)
    {
        $orderItemVariants = OrderItemVariant::where('order_item_id', $orderItem->id)->get();

        foreach ($orderItemVariants as $orderItemVariant) {
            $variant  = Variant::where('name', $orderItemVariant->variant_name)->select('id', 'name')->first();
            if ($variant) {
                $productVariant = ProductVariant::where('product_id', $product->id)
                    ->where('variant_id', $variant->id)
                    ->first();
                if ($productVariant) {

                    if ($productVariant->stock < $orderItem->quantity) {
                        return response()->json([
                            'status' => false,
                            'message' => 'stock is less than request quantity, For product ' . $product->name . ' the code ' . $product->product_code . ' current stock is ' . $productVariant->stock .
                                ' and request quantity is ' . $orderItem->quantity,
                        ]);
                    }
                    $productVariant->stock = $productVariant->stock - $orderItem->quantity;
                    $productVariant->save();
                }
            }
        }
    }


    protected function comboVariantStockIncrease($product, $orderItem)
    {
        $orderItemVariants = OrderItemVariant::where('order_item_id', $orderItem->id)->get();

        foreach ($orderItemVariants as $orderItemVariant) {
            $variant  = Variant::where('name', $orderItemVariant->variant_name)->select('id', 'name')->first();
            if ($variant) {
                $productVariant = ProductVariant::where('product_id', $product->id)
                    ->where('variant_id', $variant->id)
                    ->first();
                if ($productVariant) {

                    $productVariant->stock = $productVariant->stock + $orderItem->quantity;
                    $productVariant->save();
                }
            }
        }
    }

    protected function productStockIncrease($product, $orderItem)
    {
        $product->stock = $product->stock + $orderItem->quantity;
        $product->save();
    }
    protected function productStockDecrease($product, $orderItem)
    {
        if ($product->stock < $orderItem->quantity) {
            return response()->json([
                'status' => false,
                'message' => 'stock is less than request quantity, For product ' . $product->name . ' the code ' . $product->product_code . ' current stock is ' . $product->stock .
                    ' and request quantity is ' . $orderItem->quantity,
            ]);
        }
        $product->stock = $product->stock - $orderItem->quantity;
        $product->save();
    }
    public function OrderCourierUpdate(Request $request, $id)
    {
        $order = Order::find($id);
        if ($order) {
            $order->courier_id = $request->courier_id;
            $order->memo_no = $request->memo_no;
            if ($order->save()) {
                $courier = Courier::where('id', $order->courier_id)->first();

                return response()->json([
                    'status' => 'SUCCESS',
                    'message' => 'Courier Updated Successfully',
                    'order' => $order,
                    'courier' => $courier,
                ]);
            }
        }
    }

    public function OrderFilter(Request $request)
    {

        $orders = '';
        $paginate = $request->item ?? 10;

        if (!empty($request->start_date) && empty($request->end_date)) {
            if ($request->status != 'all') {
                $orders = Order::whereDate('created_at', '=', $request->start_date)
                    ->where('status', $request->status)
                    ->with([
                        'customer:id,name,address',
                        'createAdmin:id,name',
                        'courier',
                        'reseller:id,phone,name',
                        'orderItem.product.productImage',
                    ])
                    ->paginate($paginate);
            } else {
                $orders = Order::whereDate('created_at', '=', $request->start_date)
                    ->with([
                        'customer:id,name,address',
                        'createAdmin:id,name',
                        'courier',
                        'reseller:id,phone,name',
                        'orderItem.product.productImage',
                    ])
                    ->paginate($paginate);
            }
        }
        if (!empty($request->end_date) && !empty($request->start_date)) {
            if ($request->status != 'all') {
                $orders = Order::whereDate('created_at', '>=', $request->start_date)
                    ->whereDate('created_at', '<=', $request->end_date)
                    ->where('status', $request->status)
                    ->with([
                        'customer:id,name,address',
                        'createAdmin:id,name',
                        'courier',
                        'reseller:id,phone,name',
                        'orderItem.product.productImage',
                    ])
                    ->paginate($paginate);
            } else {
                $orders = Order::whereDate('created_at', '>=', $request->start_date)
                    ->whereDate('created_at', '<=', $request->end_date)
                    ->with([
                        'customer:id,name,address',
                        'createAdmin:id,name',
                        'courier',
                        'reseller:id,phone,name',
                        'orderItem.product.productImage',
                    ])
                    ->paginate($paginate);
            }
        }

        return response()->json([
            'status' => 'SUCCESS',
            'orders' => $orders,
            'couriers' => Courier::where('status', 1)->get(),
            'comments' => Comment::all(),
        ]);
    }

    public function OrderStatusWise(Request $request)
    {

        $paginate = $request->item ?? 20;
        $orders = Order::orderBy('id', 'DESC')
            ->where('status', $request->status)
            ->with(['customer:id,name,address', 'createAdmin:id,name', 'courier'])
            ->paginate($paginate);

        return \response()->json([
            'status' => 'SUCCESS',
            'orders' => $orders,
        ]);
    }

    public function orderSearchStatusWise(Request $request, $search)
    {
        if (strlen($search) == 11) {
            return 'ok';
            $orders = Order::where('customer_phone', 'like', '%' . $search . '%')
                ->where('status', $request->status)
                ->orderBy('id', 'DESC')
                ->with([
                    'customer:id,name,address',
                    'createAdmin:id,name',
                    'courier',
                    'reseller:id,phone,name',
                ])
                ->paginate(30);
        } else {
            $orders = Order::where('invoice_no', 'like', '%' . $search . '%')
                ->orWhere('memo_no', 'like', '%' . $search . '%')
                ->where('status', $request->status)
                ->orderBy('id', 'DESC')
                ->with([
                    'customer:id,name,address',
                    'createAdmin:id,name',
                    'courier',
                    'reseller:id,phone,name',
                ])
                ->paginate(30);
        }

        return response()->json([
            'status' => 'SUCCESS',
            'orders' => $orders,
        ]);
    }
    public function orderSearch($search)
    {
        $orders = Order::where('invoice_no', 'like', '%' . $search . '%')
            ->orWhere('customer_phone', 'like', '%' . $search . '%')
            ->orWhere('memo_no', 'like', '%' . $search . '%')
            ->orderBy('id', 'DESC')
            ->with([
                'customer:id,name,address',
                'createAdmin:id,name',
                'courier',
                'reseller:id,phone,name',
                'orderItem.product.productImage',
            ])
            ->paginate(10);

        return response()->json([
            'status' => 'SUCCESS',
            'orders' => $orders,
            'couriers' => Courier::where('status', 1)->get(),
            'comments' => Comment::all(),
        ]);
    }

    public function OrderFilterStatusWise(Request $request)
    {
        $orders = '';
        if (!empty($request->start_date) && empty($request->end_date)) {
            $orders = Order::whereDate('created_at', '=', $request->start_date)
                ->with([
                    'customer:id,name,address',
                    'createAdmin:id,name',
                    'courier',
                    'reseller:id,phone,name',
                    'orderItem.product.productImage',
                ])
                ->where('status', $request->status)
                ->paginate(30);
        }
        if (!empty($request->end_date) && !empty($request->start_date)) {
            $orders = Order::whereDate('created_at', '>=', $request->start_date)
                ->whereDate('created_at', '<=', $request->end_date)
                ->where('status', $request->status)
                ->with([
                    'customer:id,name,address',
                    'createAdmin:id,name',
                    'courier',
                    'reseller:id,phone,name',
                    'orderItem.product.productImage',
                ])
                ->paginate(30);
        }

        return response()->json([
            'status' => 'SUCCESS',
            'orders' => $orders,
            'couriers' => Courier::where('status', 1)->get(),
            'comments' => Comment::all(),
        ]);
    }

    public function orderView($id)
    {
        $orders = Order::where('id', $id)->with(['customer:id,name,address', 'courier:id,name', 'source:id,name', 'city:id,name', 'sub_city:id,name', 'approvedBy', 'wfpBy', 'orderPayment', 'createAdmin:id,name', 'packagingBy:id,name', 'pendingBy', 'shipmentBy', 'deliveredBy', 'returnBy', 'cancelBy', 'reseller:id,name,phone'])->first();

        $order_items = OrderItem::where('order_id', $orders->id)->with(['product.productVariant.variant.attribute', 'attribute', 'variant', 'variants', 'product.productImage'])->get();
        // Collect unique variant attributes
        $uniqueVariantAttributes = [];
        foreach ($order_items as $order) {
            foreach ($order->variants as $variant) {
                $uniqueVariantAttributes[$variant->variant_atribute] = true; // Store unique attributes
            }
        }
        // Get the unique attribute keys
        $variantHeaders = array_keys($uniqueVariantAttributes);
        return response()->json([
            'status' => 'SUCCESS',
            'order' => $orders,
            'variantHeaders' => $variantHeaders,
            'items' => $order_items,
        ]);
    }


    public function orderEdit($id)
    {
        $order = Order::where('id', $id)->with(['customer:id,name,address', 'courier', 'city', 'sub_city', 'approvedBy', 'orderPayment', 'createAdmin:id,name', 'pendingBy', 'shipmentBy', 'deliveredBy', 'returnBy', 'cancelBy', 'reseller:id,name,phone'])->first();
        $order->update_admin_id = session()->get('admin')['id'];
        $order->save();
        $order_items = OrderItem::where('order_id', $order->id)->with(['product.productVariant.variant.attribute', 'attribute', 'variant', 'product.productImage'])->get();
        return response()->json([
            'status' => 'SUCCESS',
            'order' => $order,
            'items' => $order_items,
        ]);
    }





    public function bulkStatusUpdate($id, $status)
    {
        try {
            DB::beginTransaction();
            $order_id = explode(',', $id);
            $orders = Order::whereIn('id', $order_id)->get();
            foreach ($orders as $order) {
                if ($status == 2) {
                    $order->pending_admin_id = session()->get('admin')['id'];
                    $order->pending_date = Carbon::now();
                }
                if ($status == 3) {
                    $order->approved_admin_id = session()->get('admin')['id'];
                    $order->approved_date = Carbon::now();
                }
                if ($status == 6) {
                    $this->cancelAll($order);
                    $order->cancel_admin_id = session()->get('admin')['id'];
                    $order->cancel_date = Carbon::now();
                }
                if ($status == 8) {
                    $order->packaging_admin_id = session()->get('admin')['id'];
                    $order->packaging_date = Carbon::now();
                }

                if ($status == 9) {
                    $order->wfp_admin_id = session()->get('admin')['id'];
                    $order->wfp_date = Carbon::now();
                }

                if ($status == 5) {
                    $order->delivered_admin_id = session()->get('admin')['id'];
                    $order->delivery_date = Carbon::now();
                }

                if ($status == 4) {
                    $this->shipment($order->id);
                }

                if ($status == 10 && ($order->status == 5 || $order->status == 6)) {
                    $items = OrderItem::where('order_id', $order->id)->get();
                    foreach ($items as $item) {
                        /***** delete order item ****/
                        $item->delete();
                    }
                    /**** delete order ****/
                    $order->delete();
                }


                if ($status != 10) {
                    $order->status = $status;
                    $order->save();
                }
            }
            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Successfully status changed',
                'order_status' => $status,
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return sendResponseWithMessage(false, $e->getMessage());
        }
    }

    public function shipmentAll(Order $order)
    {
        $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();
        if ($setting && $setting->is_pre_stock == 0) {
            $order_items = OrderItem::where('order_id', $order->id)->get();
            foreach ($order_items as $order_item) {
                $product = Product::where('id', $order_item->product_id)->firstOrFail();
                //combo campaign products
                if ($product->is_combo == 1) {

                    $combo_products_id  = ComboProduct::where('combo_product_id', $product->id)->select('general_product_id')->pluck('general_product_id');

                    $combo_products     = Product::whereIn('id', $combo_products_id)->select('id', 'stock', 'product_code')->get();

                    foreach ($combo_products as $item) {

                        if ($setting->variant_wise_stock != 1) {
                            $this->productStockDecrease($item, $order_item);
                        } else {
                            $this->comboVariantStockDecrease($item, $order_item);
                        }

                        // if ($item->stock < 1) {
                        //     return response()->json('this product ' . $item->product_code . ' stock available ' . $item->stock . ' and ordered quantity is ' . $order_item->quantity);
                        // } else {
                        //     $item->stock = $item->stock - 1;
                        //     $item->save();
                        // }
                    }
                } else {
                    if ($order_item->status == 3) {
                        $product = Product::findOrFail($order_item->product_id);
                        $this->productStockIncrease($product, $order_item);
                    } else {
                        if ($setting->variant_wise_stock != 1) {
                            $this->productStockDecrease($product, $order_item);
                        } else {
                            $this->variantStockDecrease($order_item);
                        }
                    }
                }
                $order_item->purchase_price = $product->purchase_price;
                $order_item->save();
            }
        }
    }

    public function cancelAll(Order $order)
    {
        $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();

        if ($setting->is_pre_stock == 1) {
            $details = OrderItem::where('order_id', $order->id)->get();
            foreach ($details as $order_item) {
                $product = Product::where('id', $order_item->product_id)->first();
                if ($setting->variant_wise_stock != 1) {
                    $this->productStockIncrease($product, $order_item);
                } else {
                    $this->variantStockIncrease($order_item);
                }
            }
        }

        if (!empty($order->reseller_id) && $order->order_type == 4) {
            $reseller_cashbook = ResellerCashbook::where('order_id', $order->id)->get();
            if (!empty($reseller_cashbook)) {
                foreach ($reseller_cashbook as $item) {
                    $item->delete();
                }
            }
            $order->advance_paid = 0;
            $order->paid = 0;
        }

        $order->save();
    }
    /* ==================== Bulk order assigned into courier Api Start ===================== */
    public function processOrderShipment(Order $order, int $courier_id, ?string $tracking_id = null, ?string $memo_no = null)
    {
        // Check stock and update it for each order item
        $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();
        if ($setting && $setting->is_pre_stock == 0) {
            $order_items = OrderItem::where('order_id', $order->id)->get();
            foreach ($order_items as $order_item) {
                $product = Product::where('id', $order_item->product_id)->firstOrFail();

                if ($product->is_combo == 1) {
                    $combo_products_id = ComboProduct::where('combo_product_id', $product->id)
                        ->select('general_product_id')
                        ->pluck('general_product_id');
                    $combo_products = Product::whereIn('id', $combo_products_id)
                        ->select('id', 'stock', 'product_code')
                        ->get();

                    foreach ($combo_products as $item) {
                        $item->stock -= 1;
                        $item->save();
                    }
                } else {
                    if ($order_item->status == 3) {
                        $this->productStockIncrease($product, $order_item);
                    } else {
                        if ($setting->variant_wise_stock != 1) {
                            $this->productStockDecrease($product, $order_item);
                        } else {
                            $this->variantStockDecrease($order_item);
                        }

                        $order_item->purchase_price = $product->purchase_price;
                    }
                }

                $order_item->purchase_price = $product->purchase_price ?? 0;
                $order_item->save();
            }
        }

        // Update order with shipment details
        $order->courier_id = $courier_id;
        $order->memo_no = $memo_no;
        $order->tracking_id = $tracking_id;
        $order->status = 4;
        $order->shipment_admin_id = session()->get('admin')['id'];
        $order->shipment_date = Carbon::now();
        $order->save();
    }

    /* ==================== Bulk order assigned into courier Api Start ===================== */

    public function orderBulkAssignToCourier(Request $request)
    {
        $courier_api = ModelsCourierApi::first();
        if (empty($request->courier_id)) {
            return response()->json('Courier not set');
        }

        $setting = SiteConfiguration::select('is_pre_stock')->first();

        $errors = [];
        $success = [];
        foreach ($request->orders_id as $order_id) {
            try {
                DB::beginTransaction();

                $order = Order::with(['customer', 'city', 'sub_city', 'orderItem', 'redxDeliveryArea', 'courier'])->findOrFail($order_id);

                if (!empty($order->tracking_id)) {
                    throw new Exception("Already assigned to courier");
                }
                if ($setting->is_pre_stock == 0) {
                    if ($msg = $this->checkProductStock($order)) {
                        throw new Exception($msg);
                    }
                }
                if ($request->courier_id == 14) { // Steadfast
                    $data = array();

                    $item = [
                        'invoice' => strval($order->invoice_no),
                        'recipient_name' => $order->customer ? $order->customer->name : '',
                        'recipient_address' => implode(', ', array_filter([
                                                $order->customer?->address,
                                                $order->sub_city?->name,
                                                $order->city?->name,
                                            ])),
                        'recipient_phone' => strval($order->customer_phone),
                        'cod_amount' => ($order->total + $order->shipping_cost) - ($order->discount + $order->paid),
                        'note' => $order->note ?? '',
                    ];
                    array_push($data, $item);

                    $response = (new CourierApi())->store(json_encode($data), 'steadfast');

                    if ($response['data']) {
                        foreach ($response['data'] as $value) {
                            if (!empty($value['error'])) {
                                $error = json_decode($value['error'], true);
                                $error_message = is_array($error) ? ($error[0] ?? 'Unknown error') : $value['error'];
                                throw new Exception($error_message . " IN STEADFAST PANEL");
                            }
                            if ($order) {
                                $order->memo_no =  'https://steadfast.com.bd/t/' . $value['tracking_code'];
                                $order->tracking_id =  $value['tracking_code'];
                                $order->consignment_id =  $value['consignment_id'];
                                $order->courier_id = $request->courier_id;
                                $order->save();
                            }
                        }
                    } else {
                        throw new Exception("Courier API response invalid for Steadfast");
                    }
                } else if ($request->courier_id == 7) { // Pathao
                    // if ($order->city && is_null($order->city->pathao_city_id)) {
                    //     throw new Exception('The City does not have pathao city id. changed to nearest city: ' . $order->invoice_no);
                    // }
                    // if ($order->city && ($order->city && is_null($order->sub_city->pathao_zone_id))) {
                    //     throw new Exception('The Sub-City does not have any pathao zone id. changed to nearest sub-city: ' . $order->invoice_no);
                    // }

                    // if (is_null($order->city_id) || is_null($order->sub_city_id)) {
                    //     throw new Exception('City or Sub-City information is missing for order Invoice: ' . $order->invoice_no);
                    // }

                    if (strlen($order->customer->address) < 10) {
                        throw new Exception('The Customer address must be at least 10 characters. please Edit customer address for order Invoice: ' . $order->invoice_no);
                    }
                    if ($order->status  > 2) {
                        $item = [
                            'store_id' => intval($courier_api->pathao_store_id),
                            'merchant_order_id' => strval($order->invoice_no),
                            'recipient_name' => $order->customer->name,
                            'recipient_phone' => strval($order->customer_phone),
                            'recipient_address' => implode(', ', array_filter([
                                                    $order->customer?->address,
                                                    $order->sub_city?->name,
                                                    $order->city?->name,
                                                ])),
                            // 'recipient_city' => intval($order->city->pathao_city_id),
                            // 'recipient_zone' => intval($order->sub_city->pathao_zone_id),
                            'delivery_type' => 48,
                            'item_type' => 2,
                            'special_instruction' => $order->note ?? '',
                            'item_description' => '',
                            'item_quantity' => count($order->orderItem),
                            'item_weight' => 0.5,
                            'amount_to_collect' => ($order->total + $order->shipping_cost) - ($order->discount + $order->paid),

                        ];

                        $response = (new CourierApi())->store($item, 'pathao');

                        if (isset($response['code']) && ($response['code'] == 422 || $response['code'] == 401)) {
                            if ($response['code'] == 422) {
                                $firstError = collect($response['errors'])->flatten()->first();
                                throw new Exception($firstError ?? 'Pathao API validation failed');
                            } else {
                                throw new Exception('Pathao API : ' . $response['message'] . ' - Please check your Pathao API credentials.');
                            }
                        }
                        $parcel = $response['data'];
                        if ($parcel) {
                            $order->memo_no =  'https://merchant.pathao.com/tracking?consignment_id=' . $parcel['consignment_id'] . '&phone=' . $order->customer_phone;
                            $order->tracking_id = $parcel['consignment_id'];
                            $order->courier_id = $request->courier_id;
                            $order->save();
                        } else {
                            throw new Exception("Courier API failed for Pathao");
                        }
                    }
                } else if ($request->courier_id == 11) { // RedX

                    if (!$order->redxDeliveryArea && $courier_api->redx_without_area_entry == 0) {
                        throw new Exception('RedX Delivery Area information is missing for this order invoice :  ' . $order->invoice_no);
                    }

                    if ($order->customer && strlen($order->customer->address) < 1) {
                        throw new Exception('The Customer address Required!!. please Edit customer address for order Invoice: ' . $order->invoice_no);
                    }

                    if ($order->status  > 2) {
                        $item = [
                            'customer_name' => $order->customer->name,
                            'customer_phone' => strval($order->customer_phone),
                            'customer_address' => implode(', ', array_filter([
                                                    $order->customer?->address,
                                                    $order->sub_city?->name,
                                                    $order->city?->name,
                                                ])),
                            'merchant_invoice_id' => strval($order->invoice_no),
                            'cash_collection_amount' => ($order->total + $order->shipping_cost) - ($order->discount + $order->paid),
                            'parcel_weight' => 500,
                            'instruction' => $order->note ?? '',
                            'value' => ($order->total + $order->shipping_cost) - ($order->discount + $order->paid),
                            'is_closed_box' => false,
                            'parcel_details_json' => [],
                        ];
                        if ($courier_api->redx_without_area_entry == 0) {
                            $item['delivery_area'] = $order->redxDeliveryArea->area_name ?? null;
                            $item['delivery_area_id'] = $order->redxDeliveryArea->area_id ?? null;
                        }
                        $response = (new CourierApi())->store($item, 'redx');
                        if (isset($response['validation_errors'])) {
                            $redx_errors = [];
                            foreach ($response['validation_errors'] as $fieldErrors) {
                                foreach ($fieldErrors as $msg) {
                                    $redx_errors[] = $msg;
                                }
                            }
                            throw new Exception("Validation Errors: " . implode(', ', $redx_errors));
                        }

                        if (!empty($response['tracking_id'])) {
                            $order->memo_no =  'https://redx.com.bd/track-parcel/?trackingId=' . $response['tracking_id'] . '&shopId=' . $courier_api->redx_store_id;
                            $order->tracking_id = $response['tracking_id'];
                            $order->courier_id = $request->courier_id;
                            $order->save();
                        } else {
                            throw new Exception("Courier API failed for RedX");
                        }
                    }
                } else if ($request->courier_id == 4) { // Paperfly
                    if (is_null($order->city_id) || is_null($order->sub_city_id)) {
                        throw new Exception('City or Sub-City information is missing for order Invoice: ' . $order->invoice_no);
                    }
                    if (strlen($order->customer->address) < 10) {
                        throw new Exception('The Customer address must be at least 10 characters. please Edit customer address for order Invoice: ' . $order->invoice_no);
                    }

                    $orderData = [
                        "merOrderRef" => strval($order->invoice_no),
                        "pickMerchantName" => "",
                        "pickMerchantAddress" => "",
                        "pickMerchantThana" => "",
                        "pickMerchantDistrict" => "",
                        "pickupMerchantPhone" => "",
                        "productSizeWeight" => "standard",
                        "productBrief" => "",
                        "packagePrice" => ($order->total + $order->shipping_cost) - ($order->discount + $order->paid),
                        "max_weight" => "1",
                        "deliveryOption" => "regular",
                        "custname" =>  $order->customer->name ?? '',
                        "custaddress" =>  $order->customer->address ?? '',
                        "customerThana" => $order->sub_city->name ?? '',
                        "customerDistrict" => $order->city->name ?? '',
                        "custPhone" => strval($order->customer_phone),
                    ];
                    $response = (new CourierApi())->store($orderData, 'paperfly');

                    if (isset($response['response_code']) && $response['response_code'] == 400) {
                        throw new Exception($response['error']['message'] ?? 'Paperfly API Error');
                    }

                    if (!empty($response['success']['tracking_number'])) {
                        $order->memo_no = 'https://go.paperfly.com.bd/track/order/' . $response['success']['tracking_number'];
                        $order->tracking_id = $response['success']['tracking_number'];
                        $order->courier_id = $request->courier_id;
                        $order->save();
                    } else {
                        throw new Exception("Courier API failed for Paperfly");
                    }
                } else {
                    throw new Exception("Unsupported courier ID");
                }

                $res = $this->shipment($order->id);
                if (isset($res->original['success']) && !$res->original['success']) {
                    $errorMessage = $res->original['message'] ?? "Shipment failed";
                    throw new Exception("{$errorMessage}");
                }

                DB::commit();
                $success[] = $order->invoice_no;
            } catch (Throwable $th) {
                DB::rollBack();
                $errors[] = [
                    'order_id' => $order_id,
                    'invoice_no' => $order->invoice_no ?? 'N/A',
                    'error' => $th->getMessage()
                ];
            }
        }
        return response()->json([
            'success' => count($success) > 0,
            'message' => count($success) > 0
                ? (count($errors) > 0
                    ? ('Selected orders assigned to selected courier with ' . count($errors) . ' error' . (count($errors) == 1 ? ' order.' : ' orders.'))
                    : 'Selected orders assigned to selected courier.')
                : 'No orders were assigned due to errors.',
            'successful_orders' => $success,
            'failed_orders' => $errors,
        ]);
    }

    private function checkProductStock($order)
    {
        foreach ($order->orderItem as  $item) {
            $product = Product::select('id', 'name', 'stock', 'product_code')->where('id', $item->product_id)
                ->firstOrFail();

            if ($product->stock < $item->quantity) {
                return 'stock is less than request quantity, For product ' . $product->name . ' the code ' . $product->product_code . ' current stock is ' . $product->stock .
                    ' and request quantity is ' . $item->quantity;
            }
        }
    }

    public function returnAll($id)
    {
        DB::beginTransaction();
        try {
            $setting = SiteConfiguration::select('is_pre_stock', 'variant_wise_stock')->first();
            $order_id = explode(',', $id);
            $orders = Order::whereIn('id', $order_id)->get();
            foreach ($orders as $order) {
                $order->status = 7;
                $order->return_admin_id = session()->get('admin')['id'];
                $order->return_date = Carbon::now();
                $order->save();
                //re-stock product
                $order_items = OrderItem::where('order_id', $order->id)->get();
                foreach ($order_items as $order_item) {
                    $product = Product::where('id', $order_item->product_id)->firstOrFail();
                    //if combo campaign products
                    if ($product->is_combo == 1) {
                        $combo_products_id = ComboProduct::where('combo_product_id', $product->id)
                            ->select('general_product_id')
                            ->pluck('general_product_id');
                        $combo_products = Product::whereIn('id', $combo_products_id)
                            ->select('id', 'stock', 'product_code')
                            ->get();
                        foreach ($combo_products as $item) {
                            if ($setting->variant_wise_stock != 1) {
                                $this->productStockIncrease($item, $order_item);
                            } else {
                                $this->comboVariantStockIncrease($item, $order_item);
                            }
                        }
                    } else {
                        $product->stock = $product->stock + $order_item->quantity;
                        $product->save();
                    }
                }
            }
            DB::commit();
            return response()->json('Order successfully returned');
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }




    public function orderPrint($id, $print_type)
    {
        $order_id = explode(',', $id);
        $orders = Order::whereIn('id', $order_id)->with('orderItem.product:id,name,thumbnail_img,product_code', 'orderItem.variants', 'reseller')->get();

        // Collect unique variant attributes
        $uniqueVariantAttributes = [];
        foreach ($orders as $order) {
            foreach ($order->orderItem as $item) {
                foreach ($item->variants as $variant) {
                    $uniqueVariantAttributes[$variant->variant_atribute] = true; // Store unique attributes
                }
            }
        }

        // Get the unique attribute keys
        $variantHeaders = array_keys($uniqueVariantAttributes);

        foreach ($orders as $order) {
            $order->print_status = 1;
            $order->save();
        }


        if ($print_type == 'A4_PRINT') {
            return view('admin.pdf.print.invoicePrint', \compact('orders', 'variantHeaders'));
        }
        if ($print_type == 'LABEL_PRINT') {
            return view('admin.pdf.print.labelPrint', \compact('orders', 'variantHeaders'));
        }
        if ($print_type == 'POS_PRINT') {
            return view('admin.pdf.print.posPrint', \compact('orders', 'variantHeaders'));
        }
        if ($print_type == 'A5_PRINT') {
            return view('admin.pdf.print.A5Print', \compact('orders', 'variantHeaders'));
        }
    }



    public function comment(Request $request)
    {
        $order_comment = new OrderComment();
        $order_comment->order_id = $request->order_id;
        $order_comment->comment_id = $request->comment_id;
        $order_comment->comment = $request->comment;
        $order_comment->admin_id = session()->get('admin')['id'];
        $order_comment->save();

        /***** find comment *****/
        $comment = Comment::find($request->comment_id);
        if ($comment) {
            $order = Order::find($request->order_id);
            $order->comment = $comment->name . ' ' . $request->comment;
            $order->save();
        } else {
            $order = Order::find($request->order_id);
            $order->comment = $request->comment;
            $order->save();
        }
        return response()->json([
            'status' => true,
            'message' => "Comment Added"
        ]);
    }


    public function returnItem(Request $request)
    {

        $data = $request->validate([
            'item_id' => 'required|integer',
            'quantity' => 'required|integer|min:1',
        ]);

        $item = OrderItem::findOrFail($data['item_id']);
        if ($data['quantity'] >  $item->quantity) {
            return response()->json('return quantity is greater than order quantity');
        }
        //checking is already returned
        if ($item->status == 2) {
            return response()->json('Item already returned');
        }

        DB::beginTransaction();
        try {

            $product = Product::findOrFail($item->product_id);
            if ($item->admin_sale_price > 0) {
                $order_profit =  $item->price - $item->admin_sale_price;
            } else {
                $order_profit =  $item->price - $product->sale_price;
            }
            // $product->stock = $product->stock + $data['quantity'];
            // $product->save();

            //if combo campaign products
            if ($product->is_combo == 1) {
                $combo_products_id = ComboProduct::where('combo_product_id', $product->id)->select('general_product_id')->pluck('general_product_id');
                $combo_products = Product::whereIn('id', $combo_products_id)->select('id', 'stock', 'product_code')->get();
                foreach ($combo_products as $combo_product) {
                    $combo_product->stock = $combo_product->stock + $data['quantity'];
                    $combo_product->save();
                }
            } else {
                $product->stock = $product->stock + $data['quantity'];
                $product->save();
            }

            //change order item status. if all quantity return then change status to returned otherwise update quantity
            if ($data['quantity']  == $item->quantity) {
                $item->status = 2;
            } else {
                $item->quantity = $item->quantity - $data['quantity'];
            }
            $item->save();
            //update order
            $order = Order::findOrFail($item->order_id);
            $order->profit = $order->profit - ($order_profit * $data['quantity']);
            $order->total_sale_price = $order->total_sale_price - ($product->sale_price * $data['quantity']);
            $order->total = $order->total - ($item->price * $data['quantity']);
            $order->save();

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => "action success"
            ]);
        } catch (Throwable $th) {

            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage()
            ]);
        }
    }



    public function exportOrder($status, $courier_id)
    {

        $orders = Order::where('status', $status)->where('courier_id', $courier_id)->get();
        $filename = 'order-export-' . date('d-m-Y') . '-' . time() . '.csv';
        $file = fopen($filename, 'w');

        fputcsv($file, [
            'invoice' => 'invoice',
            'Name' => 'Name',
            'Phone' => 'Phone',
            'Address' => 'Address',
            'District' => 'District',
            'Area' => 'Area',
            'AreaID' => 'AreaID',
            'Division' => 'Division',
            'Cash' => 'Cash',
            'Comment' => 'Comment',
            'product selling price' => "product selling price'",
        ]);

        foreach ($orders as $k => $line) {
            $customer = Customer::where('id', $line->customer_id)->first();
            $city = City::where('id', $line->city_id)->first();
            $sub_city = SubCity::where('id', $line->sub_city_id)->first();
            $total = $line->total - $line->discount - $line->paid + $line->shipping_cost;
            $s_price = $line->total + $line->shipping_cost;
            if ($total < 0) {
                $total = 0;
            }

            fputcsv($file, [
                'Invoice' => $line->invoice_no,
                'Name' => $customer->name,
                'Phone' => $line->customer_phone,
                'Address' => $customer->address,
                'District' => $city->name ?? '',
                'Area' => $sub_city->name ?? '',
                'AreaID' => '',
                'Division' => '',
                'Cash' => $total,
                'Comment' => '',
                'product selling price' => $s_price,
            ]);
        }
        fclose($file);

        return Response::download('./' . $filename);
    }

    public function exportOrderSelectedItem($id)
    {
        $order_id = explode(',', $id);
        $orders = Order::whereIn('id', $order_id)->get();
        $filename = 'order-export-' . date('d-m-Y') . '-' . time() . '.csv';
        $file = fopen($filename, 'w');

        fputcsv($file, [
            'ActualAmount' => 'ActualAmount',
            'CollectionAmount' => 'CollectionAmount',
            'CustomerMobile' => 'CustomerMobile',
            'District' => 'District',
            'CustomerAddress' => 'CustomerAddress',
            'CustomerName' => 'CustomerName',
            'CollectionName' => 'CollectionName',
        ]);

        foreach ($orders as $k => $line) {
            $customer = Customer::where('id', $line->customer_id)->first();
            $city = City::where('id', $line->city_id)->first();
            $sub_city = SubCity::where('id', $line->sub_city_id)->first();

            $total = $line->total - $line->discount - $line->paid;
            $g_total = $line->total - $line->discount - $line->paid + $line->shipping_cost;

            $address = $customer->address;
            if (!empty($sub_city->name)) {
                $address .= ',' . $sub_city->name;
            }
            if (!empty($city->name)) {
                $address .= ',' . $city->name;
            }
            fputcsv($file, [
                'ActualAmount' => $line->total - $line->discount,
                'CollectionAmount' => $g_total,
                'CustomerMobile' => $line->customer_phone,
                'District' => $city->name ?? '',
                'CustomerAddress' => $address,
                'CustomerName' => $customer->name,
                'CollectionName' => $line->invoice_no,
            ]);
        }
        fclose($file);
        return Response::download('./' . $filename);
    }

    public function officeSaleInvoicePrint($id)
    {
        $order = Sale::with('saleItems')->findOrFail($id);
        return view('admin.pdf.print.saleInvoice', compact('order'));
    }










    public function orderProcessorsList(Request $request)
    {

        $admins = Admin::select('id', 'name', 'image')->orderBy('id', 'desc')->where('status', 1)->paginate($request->item ?? 10);
        foreach ($admins as $admin) {

            $admin->{'today_report'}  = Order::orderPressingReport($admin->id, Carbon::today()->startOfDay(), Carbon::today()->endOfDay());
            $admin->{'yesterday_report'}  = Order::orderPressingReport($admin->id, Carbon::yesterday()->startOfDay(), Carbon::yesterday()->endOfDay());
            $admin->{'this_week_report'}  = Order::orderPressingReport($admin->id, Carbon::today()->subDays('7')->startOfDay(), Carbon::today()->endOfDay());
            $admin->{'this_month_report'}  = Order::orderPressingReport($admin->id, Carbon::today()->subDays('30')->startOfDay(), Carbon::today()->endOfDay());
            $admin->{'custom_date_report'}  = !empty($request->start_date && $request->end_date) ?  Order::orderPressingReport($admin->id, $request->start_date, $request->end_date) : [];
        }

        return response()->json([
            'success' => true,
            'admins' => $admins,
        ]);
    }



    public function orderProcessorDetails(Request $request, $id)
    {

        $admin = Admin::select('id', 'name', 'image')->findOrFail($id);
        $admin->{'today_report'}  = Order::orderPressingReport($admin->id, Carbon::today()->startOfDay(), Carbon::today()->endOfDay());
        $admin->{'yesterday_report'}  = Order::orderPressingReport($admin->id, Carbon::yesterday()->startOfDay(), Carbon::yesterday()->endOfDay());
        $admin->{'this_week_report'}  = Order::orderPressingReport($admin->id, Carbon::today()->subDays('7')->startOfDay(), Carbon::today()->endOfDay());
        $admin->{'this_month_report'}  = Order::orderPressingReport($admin->id, Carbon::today()->subDays('30')->startOfDay(), Carbon::today()->endOfDay());
        $admin->{'custom_date_report'}  = !empty($request->start_date && $request->end_date) ?  Order::orderPressingReport($admin->id, $request->start_date, $request->end_date) : [];

        return response()->json([
            'success' => true,
            'admin' => $admin,
        ]);
    }



    public function exportOrderPickList($id)
    {
        $orders_id = explode(',', $id);

        // Fetch the order invoices
        $order_invoices = Order::whereIn('id', $orders_id)->select('invoice_no')->get();
        $orders = DB::table('orders')
            ->join('order_items', 'orders.id', '=', 'order_items.order_id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->leftJoin('order_item_variants', 'order_items.id', '=', 'order_item_variants.order_item_id')
            ->whereIn('orders.id', $orders_id)
            ->groupBy(
                'order_items.product_id',
                'products.thumbnail_img',
                'products.slug',
                'products.product_code',
                'products.name',
                'order_item_variants.variant_name' // সাইজ আলাদা দেখানোর জন্য
            )
            ->select(
                'order_items.product_id',
                'products.product_code',
                'products.name',
                'products.thumbnail_img',
                'products.slug',
                'order_item_variants.variant_name',
                DB::raw('SUM(order_items.quantity) as total_qty')
            )
            ->get();
        $company_info = GeneralSetting::latest()->first();

        // return $orders;
        return view('admin.pdf.orderPickList', \compact('orders', 'order_invoices', 'company_info'));
    }


    public function exportOrderForPaperFly($id)
    {

        $order_id = explode(',', $id);
        $orders = Order::whereIn('id', $order_id)->get();
        $filename = 'order-export-' . date('d-m-Y') . '-' . time() . '.csv';
        $file = fopen($filename, "w");

        fputcsv($file, array(
            'Merchant Code' => "Merchant Code",
            'Merchant Order Reference' => 'Merchant Order Reference',
            'Package Price' => "Package Price",
            'Customer Name' => "Customer Name",
            'Customer Address' => "Customer Address",
            'Customer Thana' => "Customer Thana",
            'Customer District' => "Customer District",
            'Customer Phone' => "Customer Phone"
        ));

        foreach ($orders as $k => $line) {

            $customer = Customer::where('id', $line->customer_id)->first();
            $city = City::where('id', $line->city_id)->first();
            $sub_city = SubCity::where('id', $line->sub_city_id)->first();

            $total = (($line->total - ($line->discount + $line->paid)) + $line->shipping_cost);

            $address = $customer->address;
            if (!empty($sub_city->name)) {
                $address .= ',' . $sub_city->name;
            }
            if (!empty($city->name)) {
                $address .= ',' . $city->name;
            }
            fputcsv($file, array(
                'Merchant Code' =>  'C-1-30163',
                'Merchant Order Reference' =>   $line->invoice_no,
                'Package Price' => $total,
                'Customer Name' => $customer->name,
                'Customer Address' => $address,
                'Customer Thana' => $sub_city,
                'Customer District' => $city->name,
                'Customer Phone' => $line->customer_phone
            ));
        }
        fclose($file);
        return Response::download('./' . $filename);
    }



    public function exportOrderForRedx($id)
    {
        $order_ids = explode(',', $id);

        $orders = Order::whereIn('id', $order_ids)
            ->with(['customer', 'city', 'sub_city', 'orderItem', 'redxDeliveryArea'])
            ->get();

        $generalSettings = GeneralSetting::latest()->first();

        $export_items = $orders->map(function ($order) use ($generalSettings) {
            $customer = $order->customer;
            $city = $order->city;
            $sub_city = $order->sub_city;

            $total = ($order->total - ($order->discount + $order->paid)) + $order->shipping_cost;
            $address = trim($customer->address . ', ' . ($sub_city->name ?? '') . ', ' . ($city->name ?? ''), ', ');

            return [
                'Invoice' =>  $order->invoice_no,
                'Customer Name' => $customer->name ?? '',
                'Contact No.' => $order->customer_phone ?? '',
                'Customer Address' => $address ?? '',
                'District' => ucwords($city ? $city->name : ''),
                'Area' => ucwords($sub_city ? $sub_city->name : ''),
                'Area ID' => $order->redxDeliveryArea ? $order->redxDeliveryArea->area_id : 48,
                'Division' => "Division",
                'Price' => $total,
                'Instruction' => $order->note ?? '',
                'Product Selling Price' => $total,
                'Seller Name' => $generalSettings->title ?? '',
                'Seller Phone' =>  $generalSettings->header_contact_number ?? '',
            ];
        });

        $headers = [
            'Invoice',
            'Customer Name',
            'Contact No',
            'Customer Address',
            'District',
            'Area',
            'Area ID',
            'Division',
            'Price',
            'Instruction',
            'Product Selling Price',
            'Seller Name',
            'Seller Phone'
        ];

        return (new OrdersExport($export_items->toArray(), $headers))->download('order-export-for-redx-' . date('d-m-Y') . '.xlsx');
    }



    public function exportOrderForPathao($id)
    {
        $order_ids = explode(',', $id);

        $orders = Order::whereIn('id', $order_ids)
            ->with(['customer', 'city', 'sub_city', 'orderItem'])
            ->get();

        $generalSettings = GeneralSetting::latest()->first();
        $companyName = $generalSettings->title ?? '';

        $export_items = $orders->map(function ($order) use ($companyName) {
            $customer = $order->customer;
            $city = $order->city;
            $sub_city = $order->sub_city;

            $total = ($order->total - ($order->discount + $order->paid)) + $order->shipping_cost;
            $address = trim($customer->address . ', ' . ($sub_city->name ?? '') . ', ' . ($city->name ?? ''), ', ');

            return [
                'ItemType' => 'parcel',
                'StoreName' => $companyName,
                'MerchantOrderId' => $order->invoice_no,
                'RecipientName' => $customer->name ?? '',
                'RecipientPhone' => $order->customer_phone,
                'RecipientCity' => ucwords($city->name ?? ''),
                'RecipientZone' => ucwords($sub_city->name ?? ''),
                'RecipientArea' => ucwords($sub_city->name ?? ''),
                'RecipientAddress' => $address,
                'AmountToCollect' => $total,
                'ItemQuantity' => $order->orderItem->sum('quantity'),
                'ItemWeight' => 500,
                'ItemDesc' => '',
                'SpecialInstruction' => $order->note ?? '',
            ];
        });

        $headers = [
            'ItemType',
            'StoreName',
            'MerchantOrderId',
            'RecipientName',
            'RecipientPhone',
            'RecipientCity',
            'RecipientZone',
            'RecipientArea',
            'RecipientAddress',
            'AmountToCollect',
            'ItemQuantity',
            'ItemWeight',
            'ItemDesc',
            'SpecialInstruction'
        ];

        return (new OrdersExport($export_items->toArray(), $headers))->download('order-export-for-pathao-' . date('d-m-Y') . '.xlsx');
    }


    public function exportOrderForeCourier($id)
    {

        $order_id = explode(',', $id);
        $orders = Order::whereIn('id', $order_id)->get();
        $filename = 'order-export-for-eCourier-' . date('d-m-Y') . '.csv';
        $file = fopen($filename, "w");

        fputcsv($file, array(
            'SL' => "SL",
            'CUSTOMER NAME' => "CUSTOMER NAME",
            'ADDRESS' => "ADDRESS",
            'MOBILE' => "MOBILE",
            'ACTUAL PRODUCT PRICE' => "ACTUAL PRODUCT PRICE",
            'COLLECTED AMOUNT' => "COLLECTED AMOUNT",
            'PRODUCT ID OR INVOICE NUMBER' => "PRODUCT ID",
            'INSTRUCTION' => "INSTRUCTION",
        ));

        foreach ($orders as $k => $line) {


            $customer = Customer::where('id', $line->customer_id)->firstOrFail();
            $city = City::where('id', $line->city_id)->firstOrFail();
            $sub_city = SubCity::where('id', $line->sub_city_id)->first();

            $total = (($line->total - ($line->discount + $line->paid)) + $line->shipping_cost);

            $address = $customer->address;
            if (!empty($sub_city->name)) {
                $address .= ',' . $sub_city->name;
            }
            if (!empty($city->name)) {
                $address .= ',' . $city->name;
            }
            fputcsv($file, array(
                'SL' => $k + 1,
                'CUSTOMER NAME' =>  $customer->name,
                'ADDRESS' => $address,
                'MOBILE' =>  $line->customer_phone,
                'ACTUAL PRODUCT PRICE' =>  $total - $line->shipping_cost,
                'COLLECTED AMOUNT' => $total,
                'PRODUCT ID' =>  $line->invoice_no,
                'INSTRUCTION' => '',

            ));
        }
        fclose($file);
        return Response::download('./' . $filename);
    }



    public function exportOrderForSteadfast($ids)
    {
        $order_id = explode(',', $ids);
        $orders = Order::whereIn('id', $order_id)->get();

        $export_items = [];

        foreach ($orders as $order) {
            $customer = Customer::where('id', $order->customer_id)->firstOrFail();
            $total =  ($order->total + $order->shipping_cost) - ($order->discount + $order->paid);

            $export_items[] = [
                'Invoice' => $order->invoice_no,
                'Name' => $customer->name,
                'Address' => $customer->address,
                'Phone' => $order->customer_phone,
                'Amount' => $total,
                'Paid By' => $order->paid_by,
                'Note' => $order->note ?? '',
                'Contact Name' => GeneralSetting::latest()->first()->title,
                'Contact Phone' => GeneralSetting::latest()->first()->header_contact_number,
            ];
        }

        $headers = ['Invoice', 'Name', 'Address', 'Phone', 'Amount', 'Paid By', 'Note', 'Contact Name', 'Contact Phone'];

        return (new OrdersExport($export_items, $headers))->download('order-export-for-steadfast-' . date('d-m-Y') . '.xlsx');
    }





    public function storeExchange(StoreRequest $request)
    {

        DB::beginTransaction();
        try {

            $data = $request->validated();
            //first find find the customer
            $customer = Customer::where('phone', $data['customer_phone'])->first();
            !empty($customer) ? $customer->update($data) :    $customer = Customer::storeCustomer($data);
            //order data
            $data['customer_id'] = $customer->id;
            $data['invoice_no'] = 123;
            $data['is_exchanging'] = 1;
            $data['create_admin_id'] = session()->get('admin')['id'];
            //store who is approved
            if ($data['status'] == 3) {
                $data['approved_admin_id'] = session()->get('admin')['id'];
                $data['approved_date'] = Carbon::now();
            }
            $order = Order::query()->create($data);
            //save exchange items 
            foreach ($request->exist_order_items as $item) {
                if ($item['status'] == 3) {
                    $o_item = new OrderItem();
                    $o_item->order_id = $order->id;
                    $o_item->product_id = $item['product_id'];
                    $o_item->price = $item['price'];
                    $o_item->quantity = $item['quantity'];
                    $o_item->total = $item['quantity'] * $item['price'];
                    // $o_item->variant_id=$item['variant_id'] ?? null;

                    $o_item->size = $item['variant_id'] ?? null;
                    // $o_item->color = $product['color'] ?? null;
                    // $o_item->weight = $product['weight'] ?? null;


                    $o_item->status = 3;
                    $o_item->save();
                }
            }
            //save new items
            foreach ($request->products as $item) {
                $o_item = new OrderItem();
                $o_item->order_id = $order->id;
                $o_item->product_id = $item['id'];
                $o_item->price = $item['price'];
                $o_item->quantity = $item['quantity'];
                $o_item->total = $item['quantity'] * $item['price'];
                $o_item->size = $item['variant_id'] ?? null;
                $o_item->save();

                if ($item['variant_id']) {
                    $variant = Variant::with('attribute')->where('id', $item['variant_id'])->first();
                    OrderItemVariant::create([
                        'order_item_id' => $o_item->id,
                        'variant_atribute' => $variant->attribute->name ?? null,
                        'variant_name' => $variant->name ?? null,
                    ]);
                }
            }
            /**** making 6 digit unique invoice no ****/
            if (strlen($order->id) > 3) {
                $order->invoice_no = rand(1, 9) . $order->id . rand(1, 9);
            } else if (strlen($order->id) > 2) {
                $order->invoice_no = rand(11, 99) . $order->id . rand(1, 9);
            } else if (strlen($order->id) > 1) {
                $order->invoice_no = rand(11, 99) . $order->id . rand(11, 99);
            } else {
                $order->invoice_no = rand(111, 999) . $order->id . rand(11, 99);
            }
            $order->save();

            //create a barcode for order
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }




    public function updateExchange(UpdateRequest $request, $id)
    {

        $data = $request->validated();
        $order = Order::findOrFail($id);
        if ($order->status != 1 && $order->status != 2  && $order->status != 3) {
            return response()->json('Order New Or Pending not now');
        }

        DB::beginTransaction();
        try {
            //first find find the customer
            $customer = Customer::where('phone', $data['customer_phone'])->first();
            !empty($customer) ? $customer->update($data) : $customer = Customer::storeCustomer($data);

            //update credit
            if ($order->order_type != 4 &&   $data['paid'] > 0) {
                $credit = Credit::where('order_id', $order->id)->first();
                if ($credit) {
                    $credit->amount = $data['paid'];
                    $credit->purpose = "order advance paid amount updated ";
                    $credit->comment = "order advance paid amount updated previous amount " . $credit->amount . ' updated amount ' . $data['paid'];
                    $credit->save();
                } else {
                    $balance = Balance::where('department', 'mohasagor.com')->where('id', $data['balance_id'])->first();
                    $comment = "order advance paid amount " . $data['paid'] . " the order invoice number is " . $order->invoice_no;
                    AccountService::creditStore('order advance payment', $data['paid'], $balance->id, $order->id, null, $comment);
                }
            }
            //store who is approved
            if ($data['status'] == 3 && empty($order->approved_admin_id)) {
                $data['approved_admin_id'] = session()->get('admin')['id'];
                $data['approved_date'] = Carbon::now();
            }
            //store who is update
            if ($data['status'] == 2) {
                $data['pending_admin_id'] = session()->get('admin')['id'];
                $data['pending_date'] = Carbon::now();
            }
            $data['customer_id'] = $customer->id;

            $data['is_exchanging'] = 1;

            $order->update($data);
            //update or insert items
            foreach ($request->products as $product) {
                if ($product['status'] == 1) {
                    //delete item
                    OrderItem::where('order_id', $order->id)->where('product_id', $product['id'])->orderBy('id', 'desc')->delete();
                    //store item
                    $item = new OrderItem();
                    $item->order_id = $order->id;
                    $item->product_id = $product['id'];
                    $item->price = $product['price'];
                    $item->quantity = $product['quantity'];
                    $item->total = $product['quantity'] * $product['price'];
                    $item->attribute_id = $product['attribute_id'] ?? null;
                    $item->size = $product['variant_id'] ?? null;
                    $item->save();
                }
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Order  updated successfully'
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }




    public function searchForExchange($invoice_no)
    {

        $order = Order::where('invoice_no', $invoice_no)
            ->with([
                'customer:id,name,address',
                'courier:id,name',
                'city:id,name',
                'sub_city:id,name'
            ])
            ->firstOrFail();
        $items = OrderItem::where('order_id', $order->id)
            ->with([
                'product:id,name,product_code,thumbnail_img,price',
                'orderItemVariant'
            ])
            ->get();
        return response()->json([
            'success' => true,
            'order' => $order,
            'items' => $items,
        ]);
    }



    public function exchangeItemStatus($id)
    {
        $item = OrderItem::findOrFail($id);

        try {
            if ($item->status == 3) {
                $item->status = 1;
                $item->save();
            } else {
                $item->status = 3;
                $item->save();
            }

            return response()->json([
                'success' => true,
                'status' => $item->status,
                'message' => 'set up exchanged item ',
            ]);
        } catch (Throwable $th) {

            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }

    public function returnWithCost(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            // return $request->all();
            $order = Order::findOrFail($id);
            $order->is_profit_paid = 0;
            if (intval($request->return_cost > 0)) {
                $order->paid = $request->return_cost;
                $order->return_paid = $request->return_cost;
                $comment = $order->paid . " delivery charges were paid by the customer while returning the product";
                //expense from reseller wallet
                if ($order->reseller_id) {
                    HelperService::resellerCashbookStore($order->id, $order->reseller_id, $order->paid, 1,  $comment);
                }
                AccountService::creditStore('Return Order', $order->paid, $request->balance_id, $order->id, null, $comment);
            }

            $order->save();

            //if order is reseller order. and order is returned then apply return charge from reseller cashbook
            if (!empty($order->reseller_id) && $order->order_type == 4) {
                //expense from reseller wallet
                HelperService::resellerCashbookStore($order->id, $order->reseller_id, $order->regular_shipping_cost, 0,  'order shipping cost BDT ' . $order->regular_shipping_cost);
            }
            //if order is reseller order. and order has paid amount then delete previous reseller cashbook
            if ($order->paid > 0 && !empty($order->reseller_id) && $order->order_type == 4) {
                //checking if order is paid by admin
                $credit_check = Credit::where('order_id', $order->id)->first();
                if (!$credit_check) {

                    $order = Order::findOrFail($order->id);
                    $order->paid = 0;
                    $order->advance_paid = 0;
                    $order->save();

                    $reseller_cashbook = ResellerCashbook::where('order_id', $order->id)->where('reseller_id', $order->reseller_id)->firstOrFail();
                    $reseller_cashbook->delete();
                }
            }
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Added order return cost successfully'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return sendResponseWithMessage(false, $e->getMessage());
        }
    }

    public function getOrderComments($order_id)
    {
        $order_comments = OrderComment::with(['createdBy:id,name', 'selectedComment:id,name'])->where('order_id', $order_id)->get();
        return sendResponseWithData('order_comments', $order_comments, true, 'get successfully');
    }


    public function delete($id)
    {
        try {
            DB::beginTransaction();
            $order = Order::findOrFail($id);
            $items = OrderItem::where('order_id', $order->id)->get();
            foreach ($items as $item) {
                /***** delete order item ****/
                $item->delete();
            }
            /**** delete order ****/
            $order->delete();
            DB::commit();
            return sendResponseWithMessage(true, 'Order deleted successfully');
        } catch (Exception $e) {
            DB::rollback();
            return sendResponseWithMessage(false, $e->getMessage());
        }
    }



    public function orderSummary($id)
    {
        $transaction = ResellerPaymentTransaction::with('bank.resellerPaymentMethod', 'reseller')->findOrFail($id);
        $transaction_details = ResellerPaymentTransactionDetail::with('order', 'paymentTransaction')->where('reseller_payment_transaction_id', $transaction->id)->get();
        $cashbook_transaction_details = ResellerCashbookPaymentTransactionDetail::with('reseller_cashbook')->where('reseller_payment_transaction_id', $transaction->id)->get();
        $general_setting = GeneralSetting::latest()->first();
        $reseller = Reseller::findOrFail($transaction->reseller_id);
        $reseller_payment_method = ResellerPaymentMethod::with('bank')->where('reseller_id', $reseller->id)->first();
        return response()->json([
            "success" => true,
            "transaction_details" => $transaction_details,
            'transaction' => $transaction,
            'general_setting' => $general_setting,
            'reseller_payment_method' => $reseller_payment_method,
            'cashbook_transaction_details' => $cashbook_transaction_details,
        ]);
    }



    public function downloadOrderSummary($id)
    {
        $transaction = ResellerPaymentTransaction::with('bank.resellerPaymentMethod')->findOrFail($id);
        $transaction_details = ResellerPaymentTransactionDetail::with('order', 'paymentTransaction')->where('reseller_payment_transaction_id', $transaction->id)->get();
        $company_info = GeneralSetting::latest()->first();
        return view('reseller.pdf.transaction_details', compact('transaction_details', 'company_info'));
    }


    public function printOrderSummary($id)
    {
        $transaction = ResellerPaymentTransaction::with('bank.resellerPaymentMethod')->findOrFail($id);
        $transaction_details = ResellerPaymentTransactionDetail::with('order', 'paymentTransaction')->where('reseller_payment_transaction_id', $transaction->id)->get();
        $company_info = GeneralSetting::latest()->first();
        return view('reseller.pdf.transaction_details_print', compact('transaction_details', 'company_info'));
    }



    public function customerOrderInfo(Request $request)
    {
        $courier_api = ModelsCourierApi::first();
        $steadfast_response = Http::withHeaders([
            'Api-Key' => $courier_api->steadfast_api_key,
            'Secret-Key' => $courier_api->steadfast_secret_key,
            'Content-Type' => 'application/json'
        ])->get('https://portal.packzy.com/api/v1/fraud_check/' . $request->customer_phone);


        $redx_response = Http::withHeaders([
            'API-ACCESS-TOKEN' => 'Bearer ' . $courier_api->redx_access_token,
            'Accept' => 'application/json',
        ])->get('https://openapi.redx.com.bd/v1.0.0-beta/customer/success_rate/' . $request->customer_phone);


        $pathao_response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'accept' => 'application/json',
            'Authorization' => 'Bearer ' . $courier_api->pathao_access_token,
        ])->post("https://merchant.pathao.com/api/v1/user/success", [
            'phone' => $request->customer_phone,
        ]);


        $paperfly_response = Http::withHeaders([
            'paperflykey' => $courier_api->paperfly_api_key ?? '',
        ])->withBasicAuth(
            $courier_api->paperfly_username ?? '',
            $courier_api->paperfly_password ?? ''
        )->post('https://go-app.paperfly.com.bd/merchant/api/react/smart-check/list.php', [
            'search_text' => $request->customer_phone,
            'limit' => 50,
            'page' => 1,
        ]);


        return response()->json([
            'status' => true,
            'steadfast_response'    => json_decode($steadfast_response),
            'pathao_response'       => json_decode($pathao_response),
            'redx_response'         => json_decode($redx_response),
            'paperfly_response'     => json_decode($paperfly_response),
        ]);
    }


    public function partialDelivered(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        if ($order->status != 4) {
            return response()->json([
                'success' => false,
                'message' => 'Order is not in shipment status',
            ]);
        }

        $order->status = 10; // Partial Delivered
        $order->save();

        return response()->json([
            'success' => true,
            'message' => 'Order marked as partially delivered successfully',
        ]);
    }

    public function returnPending(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        if ($order->status != 4) {
            return response()->json([
                'success' => false,
                'message' => 'Order is not in shipment',
            ]);
        }

        $order->status = 11; // Return Pending
        $order->save();

        return response()->json([
            'success' => true,
            'message' => 'Order marked as return pending successfully',
        ]);
    }

    public function progressData()
    {
        $orders = Order::select('id', 'total_parcel', 'total_delivery', 'total_cancel')->get();
        return response()->json($orders);
    }




    public function resellerOrderDelivered(Request $request){
        $data = $request->validate([
            'order_id' => 'required',
            'balance_id' => 'required',
            'amount' => 'required',
            'cod_charge' => 'required',
            'regular_shipping_cost' => 'required',
        ]);


        $order = Order::findOrFail($data['order_id']);
        if ($order->status != 4) {
            return response()->json('order not shipment');
        }
        DB::beginTransaction();
        try {
            //when order delivered,then order amount is created at a new credit.....
            //if order is reseller order. then store reseller cashbook
            if (!empty($order->reseller_id) && $order->order_type == 4) {
                $reseller = Reseller::findOrFail($order->reseller_id);
                $general_setting = DB::table('general_settings')->first();
                $order_items = OrderItem::where('order_id', $order->id)->get();
                $cod_amount =  (intval($order->total) + intval($order->shipping_cost)) - (intval($order->discount) + intval($order->paid));
                $collected_amount_minus = $cod_amount - $data['amount'];


                if(empty($order->packaging_charge) && $general_setting->packaging_charge > 0){
                    HelperService::resellerCashbookStore($order->id, $reseller->id, $general_setting->packaging_charge, 0, 'packaging cost BDT ' . $general_setting->packaging_charge);
                    $order->packaging_charge = $general_setting->packaging_charge;
                }


                if($data['cod_charge'] > 0){
                    $order->cod_charge = $data['cod_charge'];
                    HelperService::resellerCashbookStore($order->id, $reseller->id, $order->cod_charge, 0, 'COD charge BDT ' . $order->cod_charge);
                }


                if($data['regular_shipping_cost'] > $order->regular_shipping_cost){
                    $shipping_amount = $data['regular_shipping_cost'] - $order->regular_shipping_cost;
                    $order->regular_shipping_cost = $data['regular_shipping_cost'];
                    $order->profit = $order->profit - ($collected_amount_minus + $shipping_amount);
                }else{
                    $order->profit = $order->profit - $collected_amount_minus;
                }
                
                
                
                
                $reselling_amount = $order->profit;
                //reseller commission inserted
                $amount = intval($reselling_amount) - intval($order->discount);
                $note = 'successful order  delivered commission received ' .$order->profit .' BDT order discount ' .$order->discount .' BDT the order invoice number is ' .$order->invoice_no;
                HelperService::resellerCashbookStore($order->id, $reseller->id, $order->profit, 1, $note);
                /* Comment start */
                $comment ='Delivered reseller order amount ' .intval($order->total) .' BDT, shipping charge ' . intval($order->shipping_cost) . ' BDT company profit ' .(intval($order->total) - $reselling_amount) .' BDT reseller commission  ' .$amount .' BDT  Order Invoice number is ' .$order->invoice_no .' paid by reseller ' .$order->paid .' BDT and discount ' .$order->discount .' BDT';
                /* Comment end */



                $c_amount =  (intval($order->total) + intval($order->shipping_cost)) - intval($order->discount) - intval($order->paid);
                AccountService::creditStore('Delivered Order', $c_amount,$request->balance_id,$order->id,null,$comment);

                //debit credit for reseller paid amount
                if($order->paid > 0 ){
                    //company expense to match calculation the paid amount of reseller order create paid
                   $debit = AccountService::storeDebit(111,$order->paid, 13 , 1, ' expense of dropshipper order paid amount ' . $order->paid .' the order invoice number is '.$order->invoice_no, null);
                }

            } else {
                $total = intval($order->total) + intval($order->shipping_cost) - (intval($order->paid) + intval($order->discount));
                if ($total > 0) {
                    $comment ='Delivered Order. Order Amount BDT ' .$total .' Order Invoice number is ' .$order->invoice_no;
                    $c_amount = $order->total + $order->shipping_cost - ($order->paid + $order->discount) - intval($order->paid);
                    AccountService::creditStore('Delivered Order', $c_amount,$request->balance_id,$order->id,null,$comment );
                }
            }


            $order->paid = $order->advance_paid + $data['amount'];
            $order->status = 5;
            $order->is_profit_paid = 0;
            $order->delivered_admin_id = session()->get('admin')['id'];
            $order->delivery_date = Carbon::now();
            $order->total_item = OrderItem::where('order_id', $order->id)->count();
            $order->save();

            //save purchase price for mohasagor profit calculation
            $order_items = OrderItem::where('order_id',$order->id)->get();
            foreach($order_items as $item){
                $product = Product::findOrFail($item->product_id);
                //combo campaign products
                if ($product->is_combo == 1) {
                    $combo_products_id = ComboProduct::where('combo_product_id', $product->id)->select('general_product_id')->pluck('general_product_id');
                    $combo_products_purchase_price = Product::whereIn('id', $combo_products_id)->sum('purchase_price');
                    $item->purchase_price = $combo_products_purchase_price;
                }else{
                    $item->purchase_price = $product->purchase_price;
                }
                $item->save();
            }

            DB::commit();
            return response()->json([
                'status' => 'SUCCESS',
                'message' => 'Order  delivered successfully',
            ]);
        } catch (Throwable $e) {
            DB::rollBack();
            LogTracker::failLog($e);
            return response()->json([
                'status' => 'FAILED',
                'message' => $e->getMessage(),
            ]);
        }
    }
}
