<?php

namespace App\Http\Controllers\Admin;

use Throwable;
use App\Models\Debit;
use App\Models\Balance;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\Supplier;
use Illuminate\Support\Str;
use App\Models\PurchaseItem;
use App\Services\LogTracker;
use App\Services\SmsService;
use Illuminate\Http\Request;
use App\Models\ProductVariant;
use App\Models\SupplierPayment;
use App\Services\HelperService;
use App\Services\AccountService;
use App\Models\SiteConfiguration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Intervention\Image\Facades\Image;

class PurchaseController extends Controller
{

    public function index(Request $request)
    {
        $paginate = $request->item ?? 10;
        $status = $request->status ?? 1;
        $purchases = Purchase::orderBy('id', 'DESC')->with('supplier', 'purchaseBy')->where('status', $status)->paginate($paginate);
        if ($purchases) {
            return response()->json([
                'status'    => 'SUCCESS',
                'purchases' => $purchases
            ]);
        }
    }

    public function variantWisePurchaseStore(Request $request)
    {

        DB::beginTransaction();

        try {


            $validated = $request->validate([
                'purchase_date' => 'required|date',
                'supplier_id'   => 'required|exists:suppliers,id',
                'supplier_invoice_no'    => 'nullable|string|max:100',
                'products'      => 'required|array|min:1',
                'total'         => 'nullable|numeric|min:0',
                'paid'          => 'nullable|numeric|min:0',
                'due'           => 'nullable|numeric',
                'status'        => 'nullable|numeric',
                'paid_by'       => 'nullable|string',
                'balance_id'    => 'nullable|exists:balances,id',
                'comment'       => 'nullable|string',
                'memo'          => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048'
            ]);

            // Create the purchase record
            $purchase = new Purchase();

            $purchase->supplier_id         = $request->supplier_id;
            $purchase->supplier_invoice_no = $request->supplier_invoice_no ?? (rand(1, 9) . time() . rand(1, 9));

            if ($request->invoice_no) {
                $purchase->invoice_no      = $request->invoice_no;
            } else {
                $purchase->invoice_no      = rand(1, 9) . $purchase->id . rand(1, 9);
            }

            $purchase->total               = $request->total;
            $purchase->paid                = $request->paid ?? 0;
            $purchase->status              = $request->status ?? 1;
            $purchase->comment             = $request->comment ?? null;
            $purchase->purchase_date       = $request->purchase_date;
            $purchase->balance_id          = $request->balance_id;
            $purchase->purchase_by         = auth('admin')->id();


            Log::info('Purchase created', [
                'purchase_id'   => $purchase->id,
                'details'       => $purchase->toArray()
            ]);


            // Handle memo/image upload

            if ($request->memo) {
                $purchase_memo = 'memo-' . time() . '.jpg';
                Image::make(file_get_contents($request->memo))->save(public_path('storage/images/purchase_memo/') . $purchase_memo);
                $purchase->file = 'images/purchase_memo/' . $purchase_memo;
            }

            $purchase->save();
            // return $purchase;
            // Process products
            foreach ($request->products as $item) {

                // Create purchase item record
                $p_item                  = new PurchaseItem();
                $p_item->purchase_id     = $purchase->id;
                $p_item->product_id      = $item['product_id'];
                $p_item->variant_id      = $item['variant']['id'] ?? null;
                $p_item->price           = $item['price'];
                $p_item->stock           = $item['qty'];
                $p_item->total           = $item['price'] * $item['qty'];

                $p_item->save();

                $this->updateInventory($item['product_id'], $item['variant']['id'] ?? null, $item['qty'], $item['price']);
            }

            // Record supplier payment if paid amount > 0
            if ($request->paid > 0) {
                $this->processPayment($request->supplier_id, $request->paid, $request->purchase_date, $request->balance_id, $purchase->id);
            }

            $this->sendPurchaseNotification($request->supplier_id, $request->total, $request->invoice_no);

            DB::commit();

            return response()->json([
                'status'   => 'SUCCESS',
                'message'  => 'New purchase added successfully',
                'purchase' => $purchase
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status'  => 'FAILED',
                'message' => 'Failed to add purchase: ' . $th->getMessage(),
            ], 500);
        }
    }


    protected function updateInventory($productId, $variantId, $quantity, $price)
    {
        // return $variantId;
        if ($variantId) {

            $variant = ProductVariant::where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->first();

            if ($variant) {
                $variant->increment('stock', $quantity);
            }
        } else {

            $product = Product::findOrFail($productId);

            if ($product) {
                $product->increment('stock', $quantity);
                $product->purchase_price = $price;
                $product->save();
            }
        }
    }

    protected function processPayment($supplierId, $amount, $date, $balance_id, $purchaseId)
    {
        // Supplier Payment: update or create
        SupplierPayment::updateOrCreate(
            ['purchase_id' => $purchaseId],
            [
                'supplier_id' => $supplierId,
                'amount'      => $amount,
                'date'        => $date,
                'paid_by'     => $balance_id,
            ]
        );

        // Debit: update or create
        Debit::updateOrCreate(
            ['purchase_id' => $purchaseId],
            [
                'purpose'         => 9,
                'debit_from'      => $balance_id,
                'balance_id'      => $balance_id,
                'amount'          => $amount,
                'is_expense'      => 0,
                'comment'         => 'Product Purchase Payment',
                'signature'       => null,
                'date'            => $date,
                'insert_admin_id' => session()->get('admin')['id'],
            ]
        );
    }



    // Helper method to send notifications
    protected function sendPurchaseNotification($supplierId, $amount, $invoiceNo)
    {
        $supplier = Supplier::find($supplierId);

        if ($supplier) {
            try {
                $smsResult = (new SmsService())->sendNewPurchaseMessage(
                    $supplier,
                    $amount,
                    $invoiceNo
                );
                Log::info('Purchase notification sent', [
                    'supplier_id' => $supplierId,
                    'sms_result' => $smsResult
                ]);
            } catch (\Exception $e) {
                Log::error('Failed to send notification', [
                    'error' => $e->getMessage(),
                    'supplier_id' => $supplierId
                ]);
            }
        }
    }

    protected function hasNegativeStockConflict($purchaseId, $newProducts)
    {
        foreach ($newProducts as $item) {
            $productId = $item['product_id'];
            $variantId = $item['variant']['id'] ?? null;
            $newQty    = $item['qty'];

            $oldItem = PurchaseItem::where('purchase_id', $purchaseId)
                ->where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->first();

            $oldQty = $oldItem ? $oldItem->stock : 0;
            $qtyDiff = $newQty - $oldQty;

            if ($variantId) {
                $variant = ProductVariant::with(['product', 'variant'])->find($variantId);
                if ($variant && ($variant->stock + $qtyDiff) < 0) {
                    $productCode = $variant->product->product_code ?? 'N/A';
                    return "Stock would go negative for variant: {$variant->variant->name} (Product Code: {$productCode})";
                }
            } else {
                $product = Product::find($productId);
                if ($product && ($product->stock + $qtyDiff) < 0) {
                    return "Stock would go negative for product: {$product->name} (Product Code: {$product->product_code})";
                }
            }
        }

        return null;
    }


    public function store(Request $request)
    {

        $this->validate($request, [
            'supplier_id' => 'required',
        ]);

        DB::beginTransaction();
        try {
            $purchase = new Purchase();
            $purchase->supplier_id = $request->supplier_id;
            if ($request->invoice_no) {
                $purchase->invoice_no = $request->invoice_no;
            } else {
                $purchase->invoice_no = rand(1, 9) . $purchase->id . rand(1, 9);
            }
            $purchase->total = $request->total;
            $purchase->paid = $request->paid ?? 0;
            $purchase->status = $request->status;
            $purchase->comment = $request->comment ?? null;
            $purchase->purchase_date = $request->purchase_date;
            $purchase->balance_id = $request->balance_id;
            $purchase->purchase_by = session()->get('admin')['id'];
            if ($request->memo) {
                $purchase_memo = 'memo-' . time() . '.jpg';
                Image::make(file_get_contents($request->memo))->save(public_path('storage/images/purchase_memo/') . $purchase_memo);
                $purchase->file = 'images/purchase_memo/' . $purchase_memo;
            }
            $purchase->save();


            if (!empty($request->products)) {
                //save the purchase item
                foreach ($request->products as $item) {
                    $product = Product::where('id', $item['product_id'])->first();
                    $product->stock = $product->stock + $item['quantity'];
                    $product->purchase_price = $item['price'];
                    $product->save();
                    $p_item = new PurchaseItem();
                    $p_item->purchase_id = $purchase->id;
                    $p_item->product_id = $item['product_id'];
                    $p_item->price = $item['price'];
                    $p_item->stock = $item['quantity'];
                    $p_item->save();
                }
            }

            //save a supplier paid amount
            if ($purchase->paid > 0) {
                $supplier_payment = new SupplierPayment();
                $supplier_payment->supplier_id = $request->supplier_id;
                $supplier_payment->amount = $request->paid;
                $supplier_payment->date = $request->purchase_date;
                $supplier_payment->paid_by = $request->balance_id;
                $supplier_payment->purchase_id = $purchase->id;
                $supplier_payment->save();
            }

            //create a debit
            if ($request->paid > 0) {
                $comment = "Product Purchase";
                if (empty($request->products)) {
                    $comment = "Fabrics Purchase";
                }
                $debit = new Debit();
                $debit->purpose = 9;
                $debit->balance_id = $request->balance_id;
                $debit->amount = $request->paid;
                $debit->comment = "'$comment'.Paid Amount '$request->paid'";
                $debit->date = $request->purchase_date;
                $debit->insert_admin_id = session()->get('admin')['id'];
                $debit->purchase_id = $purchase->id;
                $debit->save();
            }


            //send message to supplier
            $supplier = Supplier::findOrFail($request->supplier_id);
            (new SmsService())->sendNewPurchaseMessage($supplier, $request->total, $request->invoice_no);
            DB::commit();
            return response()->json([
                'status' => 'SUCCESS',
                'message' => 'new purchase added'
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }


    public function details($id)
    {
        $site_Config = SiteConfiguration::select('variant_wise_stock')->first();

        $purchase = Purchase::find($id);
        $detail = collect();
        $purchaseItems = collect();

        if ($site_Config->variant_wise_stock != 1) {
            $details = PurchaseItem::where('purchase_id', $id)
                ->with(['product:id,name,product_code'])->get();
        } else {
            $purchaseItems = PurchaseItem::with([
                'product:id,name,product_code',
                'variant:id,name',
                'purchaseItemsForProduct' => function ($query) {
                    $query->select('id', 'purchase_id', 'product_id', 'variant_id', 'price', 'stock')
                        ->with('variant:id,name');
                }
            ])
                ->where('purchase_id', $purchase->id)
                ->select('product_id', 'purchase_id')
                ->groupBy('product_id', 'purchase_id')
                ->get()
                ->each(function ($item) {
                    // Access preloaded relations
                    $item->product = $item->product;
                    $item->variants = $item->purchaseItemsForProduct;
                    unset($item->purchaseItemsForProduct); // Clean up if needed
                });
        }


        return response()->json([
            'status' => 'SUCCESS',
            'purchase' => $purchase,
            'details' => $details ?? [],
            'purchaseItems' => $purchaseItems ?? [],
            'merchant' => Supplier::where('id', $purchase->supplier_id)->first(),
        ]);
    }

    public function  search_according_data($search)
    {
        $purchases = Purchase::where('invoice_no', 'like', '%' . $search . '%')
            ->orderBy('id', 'DESC')->with('supplier')
            ->paginate(10);

        return response()->json([
            'status' => 'OK',
            'purchases' => $purchases
        ]);
    }

    public function  according_date_wise(Request $request)
    {

        $purchases = '';
        $paginate = $request->item ?? 10;
        if (!empty($request->start_date) && empty($request->end_date)) {
            $purchases = Purchase::whereDate('created_at', '=', $request->start_date)->with('supplier')->paginate($paginate);
        } elseif (!empty($request->end_date) && !empty($request->start_date)) {
            $purchases = Purchase::whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->with('supplier')->paginate($paginate);
        } else {
            $purchases = Purchase::whereDate('created_at', '=', $request->end_date)->with('supplier')->paginate($paginate);
        }
        return response()->json([
            'status' => 'OK',
            'purchases' => $purchases

        ]);
    }

    public function uploadFile(Request $request)
    {
        $purchase = Purchase::where('id', $request->id)->first();
        if ($request->hasFile('file')) {
            $path = $request->file('file')->store('file/memo', 'public');
            $purchase->file = $path;
            $purchase->save();
            return response()->json('ok');
        }
    }

    public function variantWisePurchaseUpdate(Request $request, $id)
    {
        $request->validate([
            'supplier_id'   => 'required|exists:suppliers,id',
            'products'      => 'required|array|min:1',
            'purchase_date' => 'required|date',
            'total'         => 'nullable|numeric|min:0',
            'paid'          => 'nullable|numeric|min:0',
            'due'           => 'nullable|numeric',
            'status'        => 'nullable|numeric',
            'paid_by'       => 'nullable|string',
            'balance_id'    => 'nullable|exists:balances,id',
            'comment'       => 'nullable|string',
            'memo'          => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048'
        ]);

        // ✅ Check for potential negative stock
        $errorMessage = $this->hasNegativeStockConflict($id, $request->products);
        if ($errorMessage) {
            return response()->json([
                'status'  => 'FAILED',
                'message' => $errorMessage
            ], 422);
        }

        DB::beginTransaction();

        try {

            $purchase = Purchase::findOrFail($id);

            $purchase->update([
                'supplier_id'         => $request->supplier_id,
                'invoice_no'          => $request->invoice_no ?? $purchase->invoice_no,
                'supplier_invoice_no' => $request->supplier_invoice_no ?? $purchase->supplier_invoice_no,
                'total'               => $request->total,
                'paid'                => $request->paid ?? 0,
                'status'              => $request->status ?? 1,
                'comment' => $request->has('comment') ? $request->input('comment') : $purchase->comment,
                'purchase_date'       => $request->purchase_date,
                'balance_id'          => $request->balance_id,
                'purchase_by'         => auth('admin')->id(),
            ]);

            // 🧾 Save new memo (if uploaded)
            if ($request->hasFile('memo')) {
                $purchase_memo = 'memo-' . time() . '.' . $request->memo->extension();
                Image::make(file_get_contents($request->memo))
                    ->save(public_path('storage/images/purchase_memo/') . $purchase_memo);
                $purchase->file = 'images/purchase_memo/' . $purchase_memo;
                $purchase->save();
            }

            // 🧮 Step 1: Reverse stock from old items
            $oldItems = PurchaseItem::where('purchase_id', $purchase->id)->get();
            foreach ($oldItems as $oldItem) {
                $this->updateInventory($oldItem->product_id, $oldItem->variant_id, -$oldItem->stock, $oldItem->price);
            }

            // 🔁 Step 2: Delete old items
            PurchaseItem::where('purchase_id', $purchase->id)->delete();

            // 🆕 Step 3: Insert new items and update stock
            foreach ($request->products as $item) {
                $p_item = new PurchaseItem([
                    'purchase_id' => $purchase->id,
                    'product_id'  => $item['product_id'],
                    'variant_id'  => $item['variant']['id'] ?? null,
                    'price'       => $item['price'],
                    'stock'       => $item['qty'],
                    'total'       => $item['price'] * $item['qty'],
                ]);
                $p_item->save();

                $this->updateInventory($item['product_id'], $item['variant']['id'] ?? null, $item['qty'], $item['price']);
            }

            // 💵 Payment processing
            if ($request->paid > 0) {
                $this->processPayment($request->supplier_id, $request->paid, $request->purchase_date, $request->balance_id, $purchase->id);
            }

            DB::commit();

            return response()->json([
                'status'   => 'SUCCESS',
                'message'  => 'Purchase updated successfully.',
                'purchase' => $purchase
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'status'  => 'FAILED',
                'message' => 'Failed to update purchase: ' . $e->getMessage()
            ], 500);
        }
    }


    public function productPurchaseUpdate(Request $request, $id)
    {
        $data = $request->validate([
            'supplier_id'         => 'required|integer',
            'purchase_items'      => 'required',
            'balance_id'          => 'nullable',
            'purchase_date'       => 'nullable',
            'supplier_invoice_no' => 'nullable',
            'total'               => 'required',
            'paid'                => 'nullable',
            'comment'             => 'nullable',
            'status'              => 'required|integer',
        ]);
        $purchase = Purchase::findOrFail($id);
        DB::beginTransaction();
        try {
            //supplier previous payment and debit delete 
            if ($purchase->paid  > 0) {
                $previous_payment = SupplierPayment::where('purchase_id', $purchase->id)->firstOrFail();
                //store previous payment 
                Debit::where('purchase_id', $purchase->id)->orderBy('id', 'desc')->firstOrFail()->delete();
                //delete previous payment 
                $previous_payment->delete();
            }
            //if new paid 

            if ($data['paid'] > 0) {
                AccountService::storeSupplierPayment($data['supplier_id'], $purchase->id, $data['paid'], $request->balance_id);
                AccountService::storeDebit(9, $data['paid'], $request->balance_id, 0, 'product purchase paid  ' . $request->paid . ' the purchase invoice no ' . $purchase->invoice_no, $purchase->id);
            }
            //memo saving 
            if ($request->file) {
                $purchase_memo = 'memo-' . time() . '.jpg';
                Image::make(file_get_contents($request->file))->save(public_path('storage/images/purchase_memo/') . $purchase_memo);
                $data['file'] = 'images/purchase_memo/' . $purchase_memo;
            }

            $purchase->update($data);

            //delete and stock remove old items
            $exist_items = PurchaseItem::where('purchase_id', $purchase->id)->orderBy('id', 'desc')->get();
            foreach ($exist_items as $key => $item) {
                $product = Product::where('id', $item->product_id)->firstOrFail();
                $product->stock =  intval($product->stock) -  intval($item->stock);
                $product->save();
                $item->delete();
            }
            //save new purchase items

            foreach ($data['purchase_items'] as $item) {

                // if($item['qty'] <= 0){
                //     return response()->json([
                //         'message' => 'Unable to save because purchase qty is zero',
                //     ]);
                // }
                // $product = Product::where('id', $item['product_id'])->first();
                // if($product->stock < 0){
                //     return response()->json([
                //         'message' => 'Unable to edit because this product less than request quantity',
                //     ]);
                // }


                $product->stock = $product->stock + $item['qty'];
                $product->purchase_price = $item['price'];
                $product->save();

                $p_item = new PurchaseItem();
                $p_item->purchase_id = $purchase->id;
                $p_item->product_id = $item['product_id'];
                $p_item->price = $item['price'];
                $p_item->stock = $item['qty'];
                $p_item->save();
            }

            DB::commit();
            return HelperService::successMessage();
        } catch (Throwable $th) {
            DB::rollBack();
            LogTracker::failLog($th, session()->get('admin'));
            return HelperService::failMessage($th->getMessage());
        }
    }

    public function PurchaseECP()
    {
        return helpECP();
    }

    public function deleteProductPurchase($id)
    {
        DB::beginTransaction();
        try {
            $purchase = Purchase::findOrFail($id);
            $purchase_items = PurchaseItem::where('purchase_id', $purchase->id)->orderBy('id', 'desc')->get();
            foreach ($purchase_items as $key => $item) {
                $p_item = Product::where('id', $item->product_id)->first();

                // if($p_item->stock < 0){
                //     return response()->json([
                //         'message' => 'Unable to edit because this product less than request quantity',
                //     ]);
                // }
                if (!empty($p_item)) {
                    $p_item->stock =  intval($p_item->stock) -  intval($item->stock);
                    $p_item->save();
                }
                //delete the item
                $item->delete();
            }
            //supplier previous payment and debit delete 

            if ($purchase->paid  > 0) {
                $previous_payment = SupplierPayment::where('purchase_id', $purchase->id)->first();
                if ($previous_payment) {
                    //store previous payment 
                    Debit::where('purchase_id', $purchase->id)->orderBy('id', 'desc')->firstOrFail()->delete();
                    //delete previous payment 
                    $previous_payment->delete();
                }
            }

            //delete purchase
            $purchase->delete();

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'purchase deleted ',
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }




    public function deletePurchaseItem($id)
    {
        DB::beginTransaction();
        try {
            $purchase_item = PurchaseItem::findOrFail($id);
            //update purchase total amount
            $purchase = Purchase::findOrFail($purchase_item->purchase_id);
            $purchase->total = $purchase->total - (intval($purchase_item->price) * intval($purchase_item->stock));
            $purchase->save();
            //minus stock

            $product = Product::where('id', $purchase_item->product_id)->firstOrFail();
            // if($product->stock < 0){
            //     return response()->json([
            //         'message' => 'Unable to edit because this product less than request quantity',
            //     ]);
            // }
            $product->stock =  intval($product->stock) -  intval($purchase_item->stock);
            $product->save();
            //delete item
            $purchase_item->delete();
            DB::commit();
            return HelperService::successMessage();
        } catch (Throwable $th) {

            DB::rollBack();
            LogTracker::failLog($th, session()->get('admin'));
            return HelperService::failMessage($th->getMessage());
        }
    }




    public function getPurchase($id)
    {
        $purchase = Purchase::findOrFail($id);

        $purchase_items = PurchaseItem::where('purchase_id', $id)->with(['product:id,slug,name,thumbnail_img,product_code', 'variant:id,name'])->get();
        return response()->json([
            'success'       => true,
            'purchase'      => $purchase,
            'purchase_items' => $purchase_items,
        ]);
    }





    public function SearchProduct($search)
    {
        $products = Product::where('name', 'like', '%' . $search . '%')->orWhere('product_code', 'like', '%' . $search . '%')->where('status', 1)->paginate(12);
        return response()->json([
            'success' => true,
            'products' => $products,
        ]);
    }


    public function getPurchaseProduct($code)
    {
        $product = Product::where('product_code', $code)->first();
        return response()->json([
            'success' => true,
            'product' => $product,
        ]);
    }
}
