<?php

namespace App\Http\Controllers\Admin;

use Exception;
use Throwable;
use App\Models\Sale;
use App\Models\Credit;
use App\Models\Company;
use App\Models\Product;
use App\Models\Customer;
use App\Models\SaleItem;
use App\Services\LogTracker;
use App\Services\SmsService;
use Illuminate\Http\Request;
use App\Models\ProductVariant;
use App\Models\CompanySalePaid;
use App\Services\AccountService;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;

class SaleController extends Controller
{
    private $company = '';
    public function __construct(Request $request)
    {
        $this->middleware('admin');
    }

    public function office_sale_index(Request $request)
    {
        $item = $request->item ?? 20;
        $sales = Sale::orderBy('id', 'DESC')
            ->where('sale_type', 1)
            ->with('company')
            ->paginate($item);
        return response()->json([
            'sales' => $sales,
            'status' => 'SUCCESS',
        ]);
    }



    public function store(Request $request)
    {
        $data = $request->validate([
            'products' => 'required',
            'sale_type' => 'required|numeric',
            'total' => 'required|numeric',
            // 'paid' => 'required|numeric',
            // 'discount' => 'required|numeric',
            'paid_by' => 'nullable|integer',
            'balance_id' => 'nullable|integer',
            'partials_payment_amount' => 'nullable|integer',
            'partials_paid_by' => 'nullable',
            'company_id' => 'nullable|numeric',
            'name' => 'nullable',
            'address' => 'nullable',
            'comment' => 'nullable',
            'mobile_no' => 'nullable|digits:11',
        ]);
        DB::beginTransaction();
        try {
            $data['create_by'] = session()->get('admin')['id'];
            $data['invoice_no'] = 1000;
            $data['discount'] = $request->discount > 0 ? $request->discount : 0;
            $data['paid'] = $request->paid > 0 ? $request->paid : 0;
            $sale = Sale::query()->create($data);
            $sale->invoice_no = $sale->id + 1000;
            $sale->save();
            //save the sale item
            foreach ($data['products'] as $item) {
                //manage product stock
                $product = Product::where('id', $item['product_id'])->firstOrFail();
                if ($product->stock <= 0 || $product->stock < $item['quantity']) {
                    return response()->json([
                        'message' => 'This product ' . $product->product_code . '  available stock is ' . $product->stock . '. but ordered quantity is ' . $item['quantity'] . ' . So please stock the product first ',
                    ]);
                } else {
                    $product->stock = $product->stock - $item['quantity'];
                    $product->save();
                }
                $sale_item = new SaleItem();
                $sale_item->sale_id = $sale->id;
                $sale_item->product_id = $item['product_id'];
                $sale_item->price = $item['price'];
                $sale_item->qty = $item['quantity'];
                $sale_item->total = $item['price'] * $item['quantity'];
                $sale_item->variant_id = $item['variant_id'] ?? null;
                $sale_item->purchase_price = $product->purchase_price;
                $sale_item->save();
            }

            


            if (!empty($data['company_id'])) {
                $company = Company::where('id', $data['company_id'])->firstOrFail();
                $data['company_id'] = $company->id;
                $data['name'] = $company->name;
                $data['address'] = $company->address;
                $data['mobile_no'] = $company->phone;
                $sale->update($data);
                if ($sale->paid > 0) {
                    AccountService::creditStore('company sale', $sale->paid - $data['partials_payment_amount'], $data['balance_id'], null,  $sale->id, 'company sale(' . $company->name . '), invoice no  S-' . $sale->id);

                    //inserting partial payment
                    if ($data['partials_payment_amount'] > 0) {
                        AccountService::creditStore('company sale',  $data['partials_payment_amount'], $data['partials_paid_by'], null, $sale->id, 'company sale, Partials Payment');
                    }
                }
                $due = $sale->total - ($sale->paid + $sale->discount);
                (new SmsService())->SendMessageToCompany($company,  $sale->total, $sale->invoice_no,  $sale->paid, $due);
            } else {
                //first search customer new or exists
                $customer = Customer::where('phone', $data['mobile_no'])->first();
                //if not customer then save, as a new customer
                if (!$customer) {
                    $customer = new Customer();
                    $customer->name = $data['name'];
                    $customer->phone = $data['mobile_no'];
                    $customer->address = $data['address'];
                    $customer->customer_type = 3;
                    $customer->password = Hash::make($request->mobile_no);
                    $customer->save();
                } else {
                    $customer->name = $data['name'];
                    $customer->address = $data['address'];
                    $customer->save();
                }


                $sale->customer_id = $customer->id;
                $sale->save();


                //send message to customer
                $amount = $sale->total - $sale->discount;
                (new SmsService())->SendMessageToCustomer($customer, $amount, $sale->id);
                //create a credit.......
                $amount = $sale->paid - $data['partials_payment_amount'];
                if ($sale->paid > 0) {
                    AccountService::creditStore('Office sale', $sale->paid, $data['balance_id'],  null, $sale->id, 'Office Sale. Invoice No  S-' . $sale->id);
                }
                if ($data['partials_payment_amount'] > 0) {
                    AccountService::creditStore('Office sale', $data['partials_payment_amount'], $data['partials_paid_by'], null,  $sale->id,  'Office sale, Partials Payment');
                }
            }

            DB::commit();
            return response()->json([
                'status' => true,
                'sale_id' => $sale->id,
                'message' => 'new sale  added',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ]);
        }
    }








    public function storeInventory(Request $request)
    {
        // return $request->all();
        $data = $request->validate([
            'products'                => 'required|array',
            'sale_type'               => 'required|numeric',
            'mobile_no'               => 'required|digits:11',
            'company_id'              => 'nullable|numeric',
            'name'                    => 'nullable|string',
            'address'                 => 'nullable|string',
            'total'                   => 'required|numeric',
            'discount'                => 'nullable|numeric',
            'paid'                    => 'nullable|numeric',
            'payable'                 => 'nullable|numeric',
            'due'                     => 'nullable|numeric',
            'balance_id'              => 'nullable|numeric',
            'partial_balance_id'      => 'nullable|numeric',
            'partials_payment_amount' => 'nullable|numeric',
            'vat'                     => 'nullable|numeric',

        ]);


        DB::beginTransaction();

        try {

            $totalPaid = $request->paid + ($request->partials_payment_amount ?? 0);
            // Create the sale record
            $sale             = new Sale();
            $sale->sale_type  = $data['sale_type'];
            $sale->mobile_no  = $data['mobile_no'];
            $sale->name       = $data['name'];
            $sale->address    = $data['address'];
            $sale->invoice_no = 1000; // Temporary value
            $sale->total      = $data['total'];
            $sale->paid       = $totalPaid;
            $sale->discount   = $data['discount'];
            // $sale->payable = $data['payable'];
            // $sale->due     = $data['due'];
            $sale->balance_id = $data['balance_id'];
            $sale->create_by  = auth()->id() ?? session()->get('admin')['id'];
            $sale->status     = 2;
            $sale->save();

            // Update invoice number
            $sale->invoice_no = $sale->id + 1000;
            $sale->save();

            foreach ($data['products'] as $item) {
                $product = Product::findOrFail($item['product_id']);

                // Handle products with variants
                if (!empty($item['variant_id'])) {
                    $productVariant = ProductVariant::find($item['variant_id']);

                    if (!$productVariant) {
                        return response()->json([
                            'status' => false,
                            'type' => 'variant_not_found',
                            'message' => "Variant '{$productVariant->variant->name}' for product '{$product->name}' has insufficient stock. Available: {$productVariant->stock}, Requested: {$item['qty']}"
                        ], 400);
                    }

                    // Check variant stock
                    if ($productVariant->stock <= 0 || $productVariant->stock < $item['qty']) {
                            return response()->json([
                                'status' => false,
                                'type' => 'insufficient_stock',
                                 'message' => "Product '{$product->name}' has insufficient stock."
                    ], 400);
                        }

                    // Update variant stock
                    $productVariant->stock -= $item['qty'];
                    $productVariant->save();
                }
                // Handle products without variants
                else {
                    // Check product stock
                    if ($product->stock <= 0 || $product->stock < $item['qty']) {
                        return response()->json([
                                'status' => false,
                                'type' => 'insufficient_stock',
                                 'message' => "Product '{$product->name}' has insufficient stock."
                    ], 400);
                    }

                    // Update product stock
                    $product->stock -= $item['qty'];
                    $product->save();
                }

                // Create sale item
                $saleItem                 = new SaleItem();
                $saleItem->sale_id        = $sale->id;
                $saleItem->product_id     = $item['product_id'];
                $saleItem->variant_id     = $item['variant_id'] ?? null;
                $saleItem->price          = $item['price'];
                $saleItem->qty            = $item['qty'];
                $saleItem->total          = $item['price'] * $item['qty'];
                $saleItem->purchase_price = $product->purchase_price;
                $saleItem->save();
            }

            if (!empty($data['company_id'])) {
                $company            = Company::where('id', $data['company_id'])->firstOrFail();
                $data['company_id'] = $company->id;
                $data['name']       = $company->name;
                $data['address']    = $company->address;
                $data['mobile_no']  = $company->phone;
                $sale->update($data);
                if ($sale->paid > 0) {
                    AccountService::creditStore('company sale', $sale->paid - $data['partials_payment_amount'], $data['balance_id'], null,  $sale->id, 'company sale(' . $company->name . '), invoice no  S-' . $sale->id);

                    //inserting partial payment
                    if ($data['partials_payment_amount'] > 0) {
                        AccountService::creditStore('company sale',  $data['partials_payment_amount'], $data['partials_paid_by'], null, $sale->id, 'company sale, Partials Payment');
                    }
                }
                $due = $sale->total - ($sale->paid + $sale->discount);
                (new SmsService())->SendMessageToCompany($company,  $sale->total, $sale->invoice_no,  $sale->paid, $due);
            } else {
                $customer = Customer::where('phone', $data['mobile_no'])->first();
                if (!$customer) {
                    $customer                = new Customer();
                    $customer->name          = $data['name'];
                    $customer->phone         = $data['mobile_no'];
                    $customer->address       = $data['address'];
                    $customer->customer_type = 3;
                    $customer->password      = Hash::make($data['mobile_no']);
                    $customer->save();
                } else {
                    $customer->name    = $data['name'];
                    $customer->address = $data['address'];
                    $customer->save();
                }

                // Record payments
                $mainPaymentAmount = $data['paid'];

                if ($request->paid > 0) {
                    AccountService::creditStore(
                        'Office sale',
                        $request->paid,
                        $request->balance_id,
                        null,
                        $sale->id,
                        'Office Sale. Invoice No S-' . $sale->id
                    );
                }

                // Handle partial payment if exists
                if ($request->partials_payment_amount > 0 && $request->partial_balance_id) {
                    AccountService::creditStore(
                        'Office sale',
                        $request->partials_payment_amount,
                        $request->partial_balance_id,
                        null,
                        $sale->id,
                        'Office sale, Partials Payment'
                    );
                }

                // Send SMS notification
                $amount = $sale->total - $sale->discount;
                (new SmsService())->SendMessageToCustomer($customer, $amount, $sale->id);
            }
            DB::commit();

            return response()->json([
                'status' => true,
                'sale_id' => $sale->id,
                'message' => 'Sale completed successfully',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // public function store_(Request $request)
    // {
    //     $data = $request->validate([
    //         'products'                => 'required',
    //         'sale_type'               => 'required|numeric', //
    //         'total'                   => 'required|numeric', //
    //         'paid'                    => 'nullable|numeric', //
    //         'discount'                => 'nullable|numeric', //
    //         'paid_by'                 => 'nullable|integer', //
    //         'balance_id'              => 'nullable|integer', //
    //         'partials_payment_amount' => 'nullable|integer', //
    //         'partials_paid_by'        => 'nullable', //
    //         'company_id'              => 'nullable|numeric', // need
    //         'name'                    => 'nullable', //
    //         'address'                 => 'nullable', //
    //         'comment'                 => 'nullable',
    //         'mobile_no'               => 'nullable|digits:11', //
    //     ]);


    //     DB::beginTransaction();
    //     try {

    //         $data['create_by']  = session()->get('admin')['id'];
    //         $data['invoice_no'] = 1000;
    //         $data['discount']   = $request->discount > 0 ? $request->discount : 0;
    //         $data['paid']       = $request->paid > 0 ? $request->paid : 0;

    //         $sale               = Sale::query()->create($data);

    //         $sale->invoice_no   = $sale->id + 1000;

    //         $sale->save();

    //         //save the sale item
    //         foreach ($data['products'] as $item) {
    //             //manage product stock
    //             $product = Product::where('id', $item['product_id'])->firstOrFail();

    //             if ($product->stock <= 0 || $product->stock < $item['quantity']) {
    //                 return response()->json([
    //                     'message' => 'This product ' . $product->product_code . '  available stock is ' . $product->stock . '. but ordered quantity is ' . $item['quantity'] . ' . So please stock the product first ',
    //                 ]);
    //             } else {
    //                 $product->stock = $product->stock - $item['quantity'];
    //                 $product->save();
    //             }

    //             $sale_item = new SaleItem();

    //             $sale_item->sale_id        = $sale->id;
    //             $sale_item->product_id     = $item['product_id'];
    //             $sale_item->price          = $item['price'];
    //             $sale_item->qty            = $item['quantity'];
    //             $sale_item->total          = $item['price'] * $item['quantity'];
    //             $sale_item->variant_id     = $item['variant_id'] ?? null;
    //             $sale_item->purchase_price = $product->purchase_price;

    //             $sale_item->save();
    //         }


    //         if (!empty($data['company_id'])) {
    //             $company = Company::where('id', $data['company_id'])->firstOrFail();
    //             $data['company_id'] = $company->id;
    //             $data['name'] = $company->name;
    //             $data['address'] = $company->address;
    //             $data['mobile_no'] = $company->phone;
    //             $sale->update($data);
    //             if ($sale->paid > 0) {
    //                 AccountService::creditStore('company sale', $sale->paid - $data['partials_payment_amount'], $data['balance_id'], null,  $sale->id, 'company sale(' . $company->name . '), invoice no  S-' . $sale->id);

    //                 //inserting partial payment
    //                 if ($data['partials_payment_amount'] > 0) {
    //                     AccountService::creditStore('company sale',  $data['partials_payment_amount'], $data['partials_paid_by'], null, $sale->id, 'company sale, Partials Payment');
    //                 }
    //             }
    //             $due = $sale->total - ($sale->paid + $sale->discount);
    //             (new SmsService())->SendMessageToCompany($company,  $sale->total, $sale->invoice_no,  $sale->paid, $due);
    //         } else {
    //             //first search customer new or exists
    //             $customer = Customer::where('phone', $data['mobile_no'])->first();
    //             //if not customer then save, as a new customer
    //             if (!$customer) {
    //                 $customer                = new Customer();
    //                 $customer->name          = $data['name'];
    //                 $customer->phone         = $data['mobile_no'];
    //                 $customer->address       = $data['address'];
    //                 $customer->customer_type = 3;
    //                 $customer->password      = Hash::make($request->mobile_no);
    //                 $customer->save();
    //             } else {
    //                 $customer->name = $data['name'];
    //                 $customer->address = $data['address'];
    //                 $customer->save();
    //             }
    //             //send message to customer
    //             $amount = $sale->total - $sale->discount;
    //             (new SmsService())->SendMessageToCustomer($customer, $amount, $sale->id);
    //             //create a credit.......
    //             $amount = $sale->paid - $data['partials_payment_amount'];
    //             if ($sale->paid > 0) {
    //                 AccountService::creditStore('Office sale', $sale->paid, $data['balance_id'],  null, $sale->id, 'Office Sale. Invoice No  S-' . $sale->id);
    //             }
    //             if ($data['partials_payment_amount'] > 0) {
    //                 AccountService::creditStore('Office sale', $data['partials_payment_amount'], $data['partials_paid_by'], null,  $sale->id,  'Office sale, Partials Payment');
    //             }
    //         }

    //         DB::commit();
    //         return response()->json([
    //             'status' => true,
    //             'sale_id' => $sale->id,
    //             'message' => 'new sale  added',
    //         ]);
    //     } catch (Exception $e) {
    //         DB::rollBack();
    //         return response()->json([
    //             'status' => false,
    //             'message' => $e->getMessage()
    //         ]);
    //     }
    // }


    public function update(Request $request, $id)
    {
        $data = $request->validate([
            'products'                => 'required',
            'sale_type'               => 'required|numeric',
            'total'                   => 'required|numeric',
            'paid_by'                 => 'nullable',
            'balance_id'              => 'nullable',
            'partials_payment_amount' => 'nullable',
            'partials_paid_by'        => 'nullable',
            // 'paid'                 => 'required|numeric',
            // 'discount'             => 'required|numeric',
            'supplier_id'             => 'nullable|numeric',
            'company_id'              => 'nullable|numeric',
            'name'                    => 'nullable',
            'address'                 => 'nullable',
            'mobile_no'               => 'nullable|digits:11',
            'company_id'              => 'nullable',
        ]);


        $sale = Sale::findOrFail($id);
        DB::beginTransaction();
        try {
            $data['discount'] = $request->discount > 0 ? $request->discount : 0;
            $data['paid'] = $request->paid > 0 ? $request->paid : 0;
            if ($data['sale_type'] == 1) {
                $customer = Customer::where('phone', $data['mobile_no'])->first();
                !empty($customer) ? $customer->update($data) : $customer = Customer::storeCustomer($data);
                $data['customer_id'] = $customer->id;
            }



            if ($data['paid'] > 0) {
                $credit = Credit::where('sale_id', $sale->id)->first();
                if (!empty($credit)) {
                    $credit->comment = 'sale paid amount updated. previous amount was BDT ' . $credit->amount . ' and sale updated paid amount is BDT ' . $data['paid'] . ' sale updated by ' . session()->get('admin')['name'];
                    $credit->amount = $data['paid'];
                    $credit->save();
                } else {
                    $comment = 'sale paid amount updated. previous amount was BDT ' . $data['paid'] . ' and sale updated paid amount is BDT ' . $data['paid'] . ' sale updated by ' . session()->get('admin')['name'];
                    AccountService::creditStore('office sale', $sale->paid, $data['balance_id'], null,  $sale->id, $comment);
                }
            }


            $data['updated_by'] = session()->get('admin')['id'];
            $sale->update($data);
            $exist_items = SaleItem::where('sale_id', $sale->id)->get();
            foreach ($exist_items as $key => $item) {
                //re-stock
                $product = Product::where('id', $item->product_id)->firstOrFail();
                $product->stock = $product->stock + $item->qty;
                $product->save();
                //delete existed item
                $item->delete();
            }

            //re-insert items
            foreach ($request->products as $product) {
                $sale_product = Product::where('id', $product['product_id'])->firstOrFail();
                if ($sale_product->stock <= 0 || $sale_product->stock < $product['quantity']) {
                    return response()->json([
                        'message' => 'This product ' . $sale_product->product_code . '  available stock is ' . $sale_product->stock . '. but ordered quantity is ' . $product['quantity'] . ' . So please stock the product first ',
                    ]);
                } else {
                    $sale_product->stock = $sale_product->stock - $product['quantity'];
                    $sale_product->save();
                }

                $details                 = new SaleItem();
                $details->sale_id        = $sale->id;
                $details->product_id     = $product['product_id'];
                $details->price          = $product['price'];
                $details->qty            = $product['quantity'];
                $details->total          = $product['quantity'] * $product['price'];
                $details->variant_id     = $product['variant_id'] ?? null;
                $details->purchase_price = $sale_product->purchase_price;
                $details->save();
            }

            DB::commit();
            return sendResponseWithMessage(true, 'Office Sale Updated');
        } catch (Throwable $th) {
            DB::rollBack();
            LogTracker::failLog($th, session()->get('admin'));
            return sendResponseWithMessage(false, $th->getMessage());
        }
    }




    public function saleEdit($id)
    {
        $sale = Sale::where('id', $id)->firstOrFail();
        $sale_items = SaleItem::where('sale_id', $sale->id)->with(['product.productVariant.variant', 'variant', 'product.productImage',])->get();
        return response()->json([
            'success' => true,
            'sale' => $sale,
            'items' => $sale_items,
        ]);
    }

    public function companySalePayment(Request $request)
    {
        $request->validate([
            'amount' => 'required|integer',
            'credit_in' => 'required',
            'date' => 'required',
        ]);
        DB::beginTransaction();
        try {
            $company             = Company::where('id', $request->company_id)->firstOrFail();
            $payment             = new CompanySalePaid();
            $payment->date       = $request->date;
            $payment->company_id = $company->id;
            $payment->amount     = $request->amount;
            $payment->credit_in  = $request->credit_in;
            $payment->comment    = $request->comment ?? null;
            $payment->save();
            AccountService::creditStore($company->name . ' paid money', $request->amount, $request->credit_in, null,  null, $request->comment);
            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Payment inserted successfully',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function CompanySaleDetails($id)
    {
        $company              = Company::findOrFail($id);
        $sales                = Sale::where('company_id', $id)->orderBy('id', 'DESC')->with('saleItems')->paginate(50);
        $paid_from_paid_table = CompanySalePaid::where('company_id', $id)->sum('amount');
        $partial_paid         = Sale::where('company_id', $id)->sum('paid');
        $paid_amount          = intval($paid_from_paid_table) + intval($partial_paid);
        $paid_records         = CompanySalePaid::with('balance')->where('company_id', $id)->orderBy('id', 'desc')->paginate(50);
        $total_purchase       = Sale::where('company_id', $id)->sum('total');
        $total_discount       = Sale::where('company_id', $id)->sum('discount');

        return response()->json([
            'sales'          => $sales,
            'status'         => 'OK',
            'company'        => $company,
            'paid_amount'    => $paid_amount,
            'paid_records'   => $paid_records,
            'total_purchase' => $total_purchase,
            'total_discount' => $total_discount,
        ]);
    }

    public function SupplierSaleDetails($id)
    {
        $sales = Sale::where('supplier_id', $id)
            ->orderBy('created_at', 'DESC')
            ->get();
        return response()->json([
            'sales' => $sales,
            'status' => 'OK',
        ]);
    }

    public function companyPayment($id)
    {
        $company = Company::findOrFail($id);
        $payments = CompanySalePaid::where('company_id', $id)
            ->orderBy('id', 'DESC')
            ->paginate(50);
        return response()->json([
            'payments' => $payments,
            'status' => 'OK',
            'company' => $company,
        ]);
    }

    public function show($id)
    {
        $sale = Sale::where('id', $id)
            ->with('company', 'create')
            ->firstOrFail();
        $sale_item = SaleItem::where('sale_id', $sale->id)->with(['product.productVariant.variant', 'variant', 'product.productImage',])->get();

        return response()->json([
            'sale' => $sale,
            'items' => $sale_item,
            'status' => 'SUCCESS',
        ]);
    }

    public function paid($id)
    {
        $sale = Sale::findOrFail($id);
        $sale->status = 2;
        if ($sale->save()) {
            //make comment
            $paid_by = '';
            if ($sale->paid_by == 1) {
                $paid_by = 'Cash';
            } elseif ($sale->paid_by == 2) {
                $paid_by = 'bKash';
            } else {
                $paid_by = 'Bank';
            }

            $comment =
                'company sale.created at ' .
                date_format($sale->created_at, 'Y-m-d') .
                ' and paid date ' .
                date_format($sale->updated_at, 'Y-m-d') .
                '. Amount BDT ' .
                $sale->total .
                ' and paid by ' .
                $paid_by;
            $credit                  = new Credit();
            $credit->purpose         = 'company sale';
            $credit->amount          = $sale->total;
            $credit->comment         = $comment;
            $credit->date            = date('Y-m-d');
            $credit->insert_admin_id = session()->get('admin')['id'];
            $credit->save();

            return response()->json([
                'success' => 'OK',
                'message' => 'sale paid was successfully',
            ]);
        }
    }

    public function return($id)
    {
        $sale = Sale::findOrFail($id);
        $sale->status = 3;
        $sale->save();

        $sale_itmes = SaleItem::where('sale_id', $id)->get();
        //save the sale item
        foreach ($sale_itmes as $item) {
            //return $item->qty;
            //manage product stock
            $product = Product::where('id', $item->product_id)->firstOrFail();
            $product->stock = $product->stock + $item->qty;
            $product->save();

            return \response()->json([
                'success' => 'OK',
                'message' => 'sale returned was successfully',
            ]);
        }
    }

    public function office_sale_search_according_data($search)
    {
        $sale_items = Sale::where('sale_type', 1)->get();
        $sales = Sale::where('id', 'like', '%' . $search . '%')
            ->orWhere('mobile_no', 'like', '%' . $search . '%')
            ->orWhere('name', 'like', '%' . $search . '%')
            ->where('sale_type', 1)
            ->orderBy('id', 'DESC')
            ->paginate(10);

        return response()->json([
            'status' => 'OK',
            'sales' => $sales,
        ]);
    }

    public function office_sale_search_according_date_wise(Request $request)
    {
        $sales = '';
        $paginate = $request->item ?? 10;
        if (!empty($request->start_date) && empty($request->end_date)) {
            $sales = Sale::whereDate('created_at', '=', $request->start_date)
                ->where('sale_type', 1)
                ->paginate($paginate);
        } elseif (!empty($request->end_date) && !empty($request->start_date)) {
            $sales = Sale::whereDate('created_at', '>=', $request->start_date)
                ->whereDate('created_at', '<=', $request->end_date)
                ->where('sale_type', 1)
                ->paginate($paginate);
        } else {
            $sales = Sale::whereDate('created_at', '=', $request->end_date)
                ->where('sale_type', 1)
                ->paginate($paginate);
        }
        return response()->json([
            'status' => 'OK',
            'sales' => $sales,
        ]);
    }

    public function company_sale_index(Request $request)
    {
        $item = $request->item ?? 20;

        $sales = Sale::orderBy('id', 'DESC')
            ->where('sale_type', 2)
            ->with('company')
            ->paginate($item);
        return response()->json([
            'sales' => $sales,
            'status' => 'SUCCESS',
        ]);
    }

    public function company_sale_search_according_data($search)
    {
        $sales = Sale::where('sale_type', 2)
            ->where('name', 'like', '%' . $search . '%')
            ->orWhere('mobile_no', 'like', '%' . $search . '%')
            ->orWhere('address', 'like', '%' . $search . '%')

            ->orderBy('id', 'DESC')
            ->paginate(10);

        return response()->json([
            'status' => 'OK',
            'sales' => $sales,
        ]);
    }

    public function company_sale_search_according_date_wise(Request $request)
    {
        $sales = '';
        $paginate = $request->item ?? 10;
        if (!empty($request->start_date) && empty($request->end_date)) {
            $sales = Sale::whereDate('created_at', '=', $request->start_date)
                ->where('sale_type', 2)
                ->paginate($paginate);
        } elseif (!empty($request->end_date) && !empty($request->start_date)) {
            $sales = Sale::whereDate('created_at', '>=', $request->start_date)
                ->whereDate('created_at', '<=', $request->end_date)
                ->where('sale_type', 2)
                ->paginate($paginate);
        } else {
            $sales = Sale::whereDate('created_at', '=', $request->end_date)
                ->where('sale_type', 2)
                ->paginate($paginate);
        }
        return response()->json([
            'status' => 'OK',
            'sales' => $sales,
        ]);
    }

    public function exchangeStore(Request $request)
    {
        DB::beginTransaction();
        try {
            if (empty($request->products)) {
                return \response()->json('Sale Product Empty');
            }

            if (empty($request->exchnage_products)) {
                return \response()->json('Exchange products Empty');
            }
            if ($request->exchange_total > $request->sale_total) {
                return \response()->json('Exchange amount can not be bigger then Sale amount');
            }
            if ($request->AmountTotal < $request->paid) {
                return \response()->json('Paid can not be bigger invoice total');
            }

            $sale             = new Sale();
            $sale->sale_type  = 1;
            $sale->paid_by    = $request->paid_by;
            $sale->name       = $request->name ?? null;
            $sale->mobile_no  = $request->mobile_no ?? null;
            $sale->address    = $request->address;
            $sale->total      = $request->AmountTotal;
            $sale->paid       = $request->paid ?? 0;
            $sale->discount   = $request->discount ?? 0;
            $sale->created_by = session()->get('admin')['id'];
            $sale->status     = 2;
            $sale->invoice_no = 1000;
            $sale->save();
            $sale->invoice_no = $sale->id + 1000;
            $sale->save();

            foreach ($request->products as $item) {
                //manage product stock
                $product = Product::where('id', $item['product_id'])->firstOrFail();
                $product->stock = $product->stock - $item['quantity'];
                $product->save();

                //save the product stock
                $sale_item                 = new SaleItem();
                $sale_item->sale_id        = $sale->id;
                $sale_item->product_id     = $item['product_id'];
                $sale_item->price          = $item['price'];
                $sale_item->qty            = $item['quantity'];
                $sale_item->total          = $item['price'] * $item['quantity'];
                $sale_item->purchase_price = $product->purchase_price;
                $sale_item->save();
            }

            foreach ($request->exchnage_products as $product) {
                //manage product stock
                $pro = Product::where('id', $product['product_id'])->firstOrFail();
                $pro->stock = $pro->stock + $product['quantity'];
                $pro->save();

                $sale_item                 = new SaleItem();
                $sale_item->sale_id        = $sale->id;
                $sale_item->product_id     = $product['product_id'];
                $sale_item->price          = $product['price'];
                $sale_item->qty            = $product['quantity'];
                $sale_item->total          = $product['price'] * $product['quantity'];
                $sale_item->purchase_price = $product->purchase_price;
                $sale_item->status         = 2;
                $sale_item->save();
            }

            //new credit
            if ($sale->paid > 0) {
                $credit                  = new Credit();
                $credit->purpose         = 'Office sale';
                $credit->amount          = $sale->paid;
                $credit->credit_in       = $sale->paid_by;
                $credit->comment         = 'Office Sale | Exchange Sale. Invoice No S-' . $sale->id;
                $credit->date            = date('Y-m-d');
                $credit->insert_admin_id = session()->get('admin')['id'];
                $credit->save();
            }
            DB::commit();
            return response()->json([
                'status' => 'SUCCESS',
                'message' => 'Exchange Sale added',
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage(),
            ]);
        }
    }



    public function delete($id)
    {
        try {
            DB::beginTransaction();
            $sale = Sale::findOrFail($id);
            $items = SaleItem::where('sale_id', $sale->id)->get();
            foreach ($items as $item) {
                /***** update product stock *****/
                $product = Product::findOrFail($item->product_id);
                $product->stock = $item->qty;
                $product->save();
                /***** delete sale item ****/
                $item->delete();
            }

            /****** find credit(if sale paid getter than 0) *****/
            if ($sale->paid > 0) {
                $credit = Credit::where('sale_id', $sale->id)->first();
                $credit->delete();
            }

            /**** delete sale ****/
            $sale->delete();
            DB::commit();
            return sendResponseWithMessage(true, 'Sale deleted successfully');
        } catch (Exception $e) {
            DB::rollback();
            return sendResponseWithMessage(false, $e->getMessage());
        }
    }
}
