<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Credit;
use App\Models\BkashToken;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Services\HelperService;
use App\Models\BkashConfiguration;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;

class BkashTokenizePaymentController extends Controller
{
    private $base_url;
    private $username;
    private $password;
    private $app_key;
    private $app_secret;

    public function __construct()
    {
        $bkash = BkashConfiguration::first();
        $this->base_url = $bkash->callbackURL;
        $this->username = $bkash->bkash_username;
        $this->password =  $bkash->bkash_password;
        $this->app_key = $bkash->bkash_app_key;
        $this->app_secret =  $bkash->bkash_app_secret;
    }
    public function authHeaders()
    {
        return array(
            'Content-Type:application/json',
            'Authorization:' . $this->grant(),
            'X-APP-Key:' . $this->app_key
        );
    }

    public function curlWithBody($url, $header, $method, $body_data)
    {
        $curl = curl_init($this->base_url . $url);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $header);
        curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $body_data);
        curl_setopt($curl, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($curl, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, FALSE);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, FALSE);
        $response = curl_exec($curl);
        curl_close($curl);
        return $response;
    }




    public function grant()
    {
        $token = BkashToken::first();

        if ($token && $token->expires_in > now()) {
            Log::info($token->id_token);
            return $token->id_token;
        }

        $header = [
            'Content-Type: application/json',
            'username:' . $this->username,
            'password:' . $this->password
        ];

        $body_data = [
            'app_key'    => $this->app_key,
            'app_secret' => $this->app_secret,
        ];

        $response = $this->curlWithBody(
            '/tokenized/checkout/token/grant',
            $header,
            'POST',
            json_encode($body_data)
        );

        $data = json_decode($response);

        if (isset($data->id_token)) {
            if ($token) {
                $token->update([
                    'id_token'   => $data->id_token,
                    'expires_in' => Carbon::now()->addMinutes(60),
                    'refresh_token' => $data->refresh_token,
                ]);
            } else {
                BkashToken::create([
                    'id_token'      => $data->id_token,
                    'expires_in'    => Carbon::now()->addMinutes(60),
                    'refresh_token' => $data->refresh_token,
                ]);
            }

            return $data->id_token;
        }

        return null; 
    }


    public function payment(Request $request)
    {
        return view('bkash.pay');
    }

    public function createPayment(Request $request)
    {
        $amount = session()->get('online_payment')['order_amount'] ?? 0;
        $order_id = session()->get('online_payment')['order_id'];
        if (!$amount || $amount < 1) {
            return response()->json(['error' => 'You should pay greater than 1 TK !!'], 400);
        }

        $header = $this->authHeaders();
        $website_url = URL::to("/");


        $body_data = array(
            'mode' => '0011',
            'payerReference' => $request->payerReference ? $request->payerReference : '1', // pass oderId or anything 
            'callbackURL' => $website_url . '/bkash-callback',
            'amount' => $amount,
            'currency' => 'BDT',
            'intent' => 'sale',
            'merchantInvoiceNumber' => $request->merchantInvoiceNumber ? $request->merchantInvoiceNumber : "Inv_" . Str::random(6)
        );

        $response = $this->curlWithBody('/tokenized/checkout/create', $header, 'POST', json_encode($body_data));

        $data = json_decode($response);
        if ($data->bkashURL) {
            $order = Order::where('id', $order_id)->first();
            $order->payment_id = $data->paymentID;
            $order->save();
            return redirect($data->bkashURL);
        }
    }

    public function executePayment($paymentID)
    {

        $header = $this->authHeaders();

        $body_data = array(
            'paymentID' => $paymentID
        );


        $response = $this->curlWithBody('/tokenized/checkout/execute', $header, 'POST', json_encode($body_data));

        return $response;
    }
    public function queryPayment($paymentID)
    {
        $header = $this->authHeaders();

        $body_data = array(
            'paymentID' => $paymentID,
        );

        $response = $this->curlWithBody('/tokenized/checkout/payment/status', $header, 'POST', json_encode($body_data));

        return $response;
    }

    public function callback(Request $request)
    {
        // return 'test';
        $allRequest = $request->all();
        // return $allRequest;
        if (isset($allRequest['status']) && $allRequest['status'] == 'success') {
            $response = $this->executePayment($allRequest['paymentID']);
            if (is_null($response)) {
                sleep(1);
                $response = $this->queryPayment($allRequest['paymentID']);
            }

            $res_array = json_decode($response, true);

            // return $res_array;


            $order = Order::where('payment_id', $allRequest['paymentID'])->first();
            if (array_key_exists("statusCode", $res_array) && $res_array['statusCode'] == '0000' && array_key_exists("transactionStatus", $res_array) && $res_array['transactionStatus'] == 'Completed') {
                $credit = new Credit();
                $credit->date = Carbon::now();
                $credit->order_id = $order->id;
                $credit->purpose = 'Order Amount Payment In Bkash';
                $credit->balance_id = 6;
                $credit->amount = $res_array['amount'];
                $credit->comment = 'Order Amount Payment In Bkash';
                $credit->insert_admin_id = null;
                $credit->save();

                $order->transaction_id = $res_array['trxID'];
                $order->payment_status = 'completed';
                $order->paid = $res_array['amount'];
                $order->payment_method = 'Bkash';
                $order->status = 3;
                $order->save();

                return redirect()->route('order.success', ['invoice_no' => $order->invoice_no]);
            }

            return redirect()->route('payment.fail', ['payment_id' => $allRequest['paymentID']])->with(['response' => $res_array['statusMessage']]);
        } else {
            return redirect()->route('payment.fail', ['payment_id' => $allRequest['paymentID']])->with(['response' => 'Payment Failed !!']);
        }
    }

    public function getRefund(Request $request)
    {
        return view('bkash.refund');
    }

    public function refundPayment(Request $request)
    {
        $order = Order::where('id', $request['order_id'])->firstOrFail();
        $body_data = array(
            'paymentID' => $order->payment_id,
            'trxID' => $order->transaction_id
        );
        $header = $this->authHeaders();
        $response = $this->curlWithBody('/tokenized/checkout/payment/refund', $header, 'POST', json_encode($body_data));

        $res_array = json_decode($response, true);

        $message = "Refund Failed !!";

        if (!isset($res_array['refundTrxID'])) {

            $body_data = array(
                'paymentID' => $order->payment_id,
                'amount' => $order->paid,
                'trxID' => $order->transaction_id,
                'sku' => 'sku',
                'reason' => $request->note
            );

            $response = $this->curlWithBody('/tokenized/checkout/payment/refund', $header, 'POST', json_encode($body_data));

            $res_array = json_decode($response, true);

            if (isset($res_array['refundTrxID'])) {

                // your database insert operation    
                // $message = "Refund successful !!.Your Refund TrxID : " . $res_array['refundTrxID'];


                //update order 
                $order->paid = 0;
                $order->payment_status = 'refund';
                $order->refund_id = $res_array['refundTrxID'];
                $order->save();

                //account update 
                $credit = Credit::where('order_id', $order->id)->first();
                $credit->comment = $request['note'];
                $credit->balance_id = 6;
                $credit->save();

                return response()->json([
                    'status' => true,
                    'response' => $response,
                    'message' => 'Successfully Refunded bkash payment'
                ]);
            }
        } else {
            $message = "Already Refunded !!.Your Refund TrxID : " . $res_array['refundTrxID'];
        }

        // return view('bkash.refund')->with([
        //     'response' => $message,
        // ]);
    }

    public function getSearchTransaction(Request $request)
    {
        return view('bkash.search');
    }

    public function searchTransaction(Request $request)
    {

        $header = $this->authHeaders();
        $body_data = array(
            'trxID' => $request->trxID,
        );

        $response = $this->curlWithBody('/tokenized/checkout/general/searchTransaction', $header, 'POST', json_encode($body_data));


        return view('bkash.search')->with([
            'response' => $response,
        ]);
    }







    public function retryPayment($payment_id)
    {
        $order = Order::where('payment_id', $payment_id)->first();

        if (!$order) {
            return redirect()->route('checkout')->with(['error' => 'Order not found!']);
        }

        $amount = $order->total + $order->shipping_cost - $order->discount;

        if ($amount < 1) {
            return redirect()->route('checkout')->with(['error' => 'Invalid payment amount!']);
        }

        $header = $this->authHeaders();
        $website_url = URL::to("/");

        $body_data = [
            'mode' => '0011',
            'payerReference' => $order->id, // Order ID
            'callbackURL' => $website_url . '/bkash-callback',
            'amount' => $amount,
            'currency' => 'BDT',
            'intent' => 'sale',
            'merchantInvoiceNumber' => "Inv_" . Str::random(6)
        ];

        $response = $this->curlWithBody('/tokenized/checkout/create', $header, 'POST', json_encode($body_data));
        $data = json_decode($response);

        if (isset($data->bkashURL)) {
            // Update order with new payment ID
            $order->payment_id = $data->paymentID;
            $order->save();

            return redirect($data->bkashURL);
        }

        return redirect()->route('checkout')->with(['error' => 'Failed to initiate payment.']);
    }
}
