<?php

namespace App\Models;

use DateTimeInterface;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Order extends Model
{

  protected $fillable = ['id', 'invoice_no', 'order_type', 'customer_id', 'customer_phone', 'city_id', 'sub_city_id', 'courier_id', 'note', 'shipping_cost', 'discount', 'paid', 'balance_id', 'total', 'status', 'print_status', 'memo_no', 'create_admin_id', 'pending_admin_id', 'pending_date', 'approved_admin_id', 'approved_date', 'shipment_admin_id', 'shipment_date', 'delivered_admin_id', 'delivery_date', 'cancel_admin_id', 'cancel_date', 'return_admin_id', 'return_date', 'assigned_admin_id', 'reseller_id', 'comment', 'coupon_id', 'coupon_disocunt', 'note', 'redx_delivery_area_id', 'order_source', 'is_exchanging', 'created_at', 'updated_at'];


  public function redxDeliveryArea()
  {
    return $this->belongsTo('App\Models\RedxDeliveryArea', 'redx_delivery_area_id');
  }

  protected function serializeDate(DateTimeInterface $date)
  {
    return $date->format('Y-m-d H:i:s');
  }



  public function customerOrders()
  {
    return $this->hasMany(Customer::class, 'id', 'customer_id')->selectRaw('count(*) as total_orders');
  }


  public function customer()
  {
    return $this->belongsTo('App\Models\Customer', 'customer_id');
  }

  public function balance()
  {
    return $this->belongsTo('App\Models\Balance', 'balance_id');
  }


  public function createAdmin()
  {
    return $this->belongsTo('App\Models\Admin', 'create_admin_id');
  }
  public function courier()
  {
    return $this->belongsTo('App\Models\Courier', 'courier_id');
  }

  public function source()
  {
    return $this->belongsTo('App\Models\OrderSource', 'order_source');
  }
  public function city()
  {
    return $this->belongsTo('App\Models\City', 'city_id');
  }
  public function sub_city()
  {
    return $this->belongsTo('App\Models\SubCity', 'sub_city_id');
  }

  public function orderItem()
  {
    return  $this->hasMany('App\Models\OrderItem', 'order_id');
  }

  public function orderPayment()
  {

    return $this->hasOne('App\Models\OrderPayment', 'order_id')->where('status', "Processing");
  }

  public function resellerOrderDetails()
  {
    return $this->hasOne('App\Models\ResellerOrderDetails', 'order_id');
  }

  public function wfpBy()
  {
    return $this->belongsTo('App\Models\Admin', 'wfp_admin_id');
  }



  public function updateAdmin()
  {
    return $this->belongsTo('App\Models\Admin', 'update_admin_id');
  }

  public function approvedBy()
  {
    return $this->belongsTo('App\Models\Admin', 'approved_admin_id');
  }
  public function orderBarcode()
  {
    return $this->hasOne('App\Models\OrderBarcode', 'order_id');
  }
  public function reseller()
  {
    return $this->belongsTo('App\Models\Reseller', 'reseller_id');
  }

  public function assignedTo()
  {
    return $this->belongsTo('App\Models\Admin', 'assigned_admin_id');
  }
  public function pendingBy()
  {
    return $this->belongsTo('App\Models\Admin', 'pending_admin_id');
  }
  public function packagingBy()
  {
    return $this->belongsTo('App\Models\Admin', 'packaging_admin_id');
  }
  public function shipmentBy()
  {
    return $this->belongsTo('App\Models\Admin', 'shipment_admin_id');
  }
  public function deliveredBy()
  {
    return $this->belongsTo('App\Models\Admin', 'delivered_admin_id');
  }
  public function returnBy()
  {
    return $this->belongsTo('App\Models\Admin', 'return_admin_id');
  }
  public function cancelBy()
  {
    return $this->belongsTo('App\Models\Admin', 'cancel_admin_id');
  }


  public static function resellerOrderCount()
  {
    $reseller_id = session()->get('reseller')['id'];
    $orders = [];
    $order = Order::select('id', 'order_type', 'status')->where('reseller_id', $reseller_id)->get();
    $orders['total'] = $order->count();
    $orders['new_order'] = $order->where('status', 1)->count();
    $orders['pending_order'] = $order->where('status', 2)->count();
    $orders['approved_order'] = $order->where('status', 3)->count();
    $orders['shipment_order'] = $order->where('status', 4)->count();
    $orders['return_order'] = $order->where('status', 7)->count();
    $orders['returned_order'] = $order->where('status', 10)->count();
    $orders['cancel_order'] = $order->where('status', 6)->count();
    $orders['packaging_order'] = $order->where('status', 8)->count();
    $orders['wfp_order'] = $order->where('status', 9)->count();
    $orders['delivered_order'] = $order->where('status', 5)->count();
    return $orders;
  }




  public static function ordersAll($request)
  {

    $paginate = $request->item ?? 10;
    if ($request->status != "all") {
      if ($request->type == "all") {
        $orders = Order::orderBy('id', 'DESC')
          ->with(['customer:id,name,address', 'createAdmin:id,name', 'approvedBy:id,name', 'courier', 'reseller:id,phone,name', 'orderItem.product:id,name,thumbnail_img,product_code', 'source:id,name'])
          ->where('status', $request->status)
          ->paginate($paginate);
      } else {
        $orders = Order::orderBy('id', 'DESC')
          ->with(['customer:id,name,address', 'createAdmin:id,name', 'approvedBy:id,name', 'courier', 'reseller:id,phone,name', 'orderItem.product:id,name,thumbnail_img,product_code', 'source:id,name'])
          ->where('status', $request->status)
          ->where('order_type', $request->type)
          ->paginate($paginate);
      }
    } else {
      if ($request->type == "all") {
        $orders = Order::orderBy('id', 'DESC')
          ->with(['customer:id,name,address', 'createAdmin:id,name', 'approvedBy:id,name', 'courier', 'reseller:id,phone,name', 'orderItem.product:id,name,thumbnail_img,product_code', 'source:id,name'])
          ->paginate($paginate);
      } else {
        $orders = Order::orderBy('id', 'DESC')->with(['customer:id,name,address', 'approvedBy:id,name', 'createAdmin:id,name', 'courier', 'reseller:id,phone,name', 'orderItem.product:id,name,thumbnail_img,product_code', 'source:id,name'])->where('order_type', $request->type)->paginate($paginate);
      }
    }
    return \response()->json([
      'status' => 'SUCCESS',
      'orders' => $orders,
      'order_count' => self::orderStatusCount($request),
      'couriers' => Courier::where('status', 1)->get(),
      'comments' => Comment::all(),
    ]);
  }


  // public static function dateWiseOrderCount($request)
  // {
  //   $order = array();
  //   $order['total'] = Order::whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['new_order'] = Order::where('status', 1)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['new_order_amount'] = Order::where('status', 1)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 1)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['pending_order'] = Order::where('status', 2)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['pending_order'] = Order::where('status', 2)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['pending_order_amount'] = Order::where('status', 2)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 2)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['packaging_order'] = Order::where('status', 8)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['packaging_order_amount'] = Order::where('status', 8)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 8)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['approved_order'] = Order::where('status', 3)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['approved_order_amount'] = Order::where('status', 3)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 3)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['shipment_order'] = Order::where('status', 4)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['shipment_order_amount'] = Order::where('status', 4)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 4)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['delivered_order'] = Order::where('status', 5)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['delivered_order_amount'] = Order::where('status', 5)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 5)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['return_order'] = Order::where('status', 7)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['return_order_amount'] = Order::where('status', 7)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 7)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['cancel_order'] = Order::where('status', 6)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['cancel_order_amount'] = Order::where('status', 6)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 6)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['wfp_order'] = Order::where('status', 9)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['wfp_order_amount'] = Order::where('status', 9)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('total') - Order::where('status', 9)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->sum('discount');
  //   $order['partial_delivered'] = Order::where('status', 10)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();
  //   $order['return_pending'] = Order::where('status', 11)->whereDate('created_at', '>=', $request->start_date)->whereDate('created_at', '<=', $request->end_date)->count();

  //   return $order;
  // }


  // public static function orderCount()
  // {
  //   $order = array();
  //   $order['total'] = Order::count();
  //   $order['total_amount'] = Order::sum('total') - Order::sum('discount');
  //   $order['new_order'] = Order::where('status', 1)->count();
  //   $order['new_order_amount'] = Order::where('status', 1)->sum('total') - Order::where('status', 1)->sum('discount');
  //   $order['pending_order'] = Order::where('status', 2)->count();
  //   $order['pending_order_amount'] = Order::where('status', 2)->sum('total') - Order::where('status', 2)->sum('discount');
  //   $order['approved_order'] = Order::where('status', 3)->count();
  //   $order['approved_order_amount'] = Order::where('status', 3)->sum('total') - Order::where('status', 3)->sum('discount');
  //   $order['shipment_order'] = Order::where('status', 4)->count();
  //   $order['shipment_order_amount'] = Order::where('status', 4)->sum('total') - Order::where('status', 4)->sum('discount');
  //   $order['delivered_order'] = Order::where('status', 5)->count();
  //   $order['delivered_order_amount'] = Order::where('status', 5)->sum('total') - Order::where('status', 5)->sum('discount');
  //   $order['cancel_order'] = Order::where('status', 6)->count();
  //   $order['cancel_order_amount'] = Order::where('status', 6)->sum('total') - Order::where('status', 6)->sum('discount');
  //   $order['return_order'] = Order::where('status', 7)->count();
  //   $order['return_order_amount'] = Order::where('status', 7)->sum('total') - Order::where('status', 7)->sum('discount');
  //   $order['packaging_order'] = Order::where('status', 8)->count();
  //   $order['packaging_order_amount'] = Order::where('status', 8)->sum('total') - Order::where('status', 8)->sum('discount');
  //   $order['wfp_order'] = Order::where('status', 9)->count();
  //   $order['wfp_order_amount'] = Order::where('status', 9)->sum('total') - Order::where('status', 9)->sum('discount');
  //   $order['partial_delivered'] = Order::where('status', 10)->count();
  //   $order['partial_delivered_amount'] = Order::where('status', 10)->sum('total') - Order::where('status', 10)->sum('discount');
  //   $order['return_pending'] = Order::where('status', 11)->count();
  //   $order['return_pending_amount'] = Order::where('status', 11)->sum('total') - Order::where('status', 11)->sum('discount');
  //   $order['incomplete_order'] = Order::where('status', 12)->count();
  //   $order['incomplete_order_amount'] = Order::where('status', 12)->sum('total') - Order::where('status', 12)->sum('discount');

  //   return $order;
  // }


  protected static function applyAdminFilter($query)
  {
    $is_super_admin = session()->get('admin')['is_super_admin'] == 1;

    if (!$is_super_admin) {
      $query->where('assigned_admin_id', session()->get('admin')['id']);
    }

    return $query;
  }

  protected static function baseQuery($status = null, $start = null, $end = null)
  {
    $query = Order::query();

    if ($status !== null) {
      $query->where('status', $status);
    }

    if ($start && $end) {
      $query->whereBetween(DB::raw('DATE(created_at)'), [$start, $end]);
    }

    return self::applyAdminFilter($query);
  }


  protected static function getOrderStats($status = null, $start = null, $end = null)
  {
    $base = self::baseQuery($status, $start, $end);

    $count = (clone $base)->count();
    $total = (clone $base)->sum('total');
    $discount = (clone $base)->sum('discount');

    return [
      'count' => $count,
      'amount' => $total - $discount,
    ];
  }


  public static function orderCount()
  {
    $statuses = [
      1 => 'new_order',
      2 => 'pending_order',
      3 => 'approved_order',
      4 => 'shipment_order',
      5 => 'delivered_order',
      6 => 'cancel_order',
      7 => 'return_order',
      8 => 'packaging_order',
      9 => 'wfp_order',
      10 => 'partial_delivered',
      11 => 'return_pending',
      12 => 'incomplete_order',
    ];

    $order = [];
    $totalStats = self::getOrderStats();
    $order['total'] = $totalStats['count'];
    $order['total_amount'] = $totalStats['amount'];

    foreach ($statuses as $status => $label) {
      $stats = self::getOrderStats($status);
      $order[$label] = $stats['count'];
      $order["{$label}_amount"] = $stats['amount'];
    }

    return $order;
  }


  public static function dateWiseOrderCount($request)
  {
    $start = $request->start_date;
    $end = $request->end_date;

    $statuses = [
      1 => 'new_order',
      2 => 'pending_order',
      3 => 'approved_order',
      4 => 'shipment_order',
      5 => 'delivered_order',
      6 => 'cancel_order',
      7 => 'return_order',
      8 => 'packaging_order',
      9 => 'wfp_order',
      10 => 'partial_delivered',
      11 => 'return_pending',
      12 => 'incomplete_order',
    ];

    $order = [];
    $totalStats = self::getOrderStats(null, $start, $end);
    $order['total'] = $totalStats['count'];
    $order['total_amount'] = $totalStats['amount'];

    foreach ($statuses as $status => $label) {
      $stats = self::getOrderStats($status, $start, $end);
      $order[$label] = $stats['count'];
      $order["{$label}_amount"] = $stats['amount'];
    }

    return $order;
  }

  public static function orderStatusCount($data)
  {

    $order = array();
    $order['total'] =  (new self)->allFilters($data)->count();
    $order['new_order'] = (new self)->statusWiseFilter(1, $data)->count();
    $order['pending_order'] = (new self)->statusWiseFilter(2, $data)->count();
    $order['approved_order'] = (new self)->statusWiseFilter(3, $data)->count();
    $order['shipment_order'] = (new self)->statusWiseFilter(4, $data)->count();
    $order['delivered_order'] = (new self)->statusWiseFilter(5, $data)->count();
    $order['cancel_order'] = (new self)->statusWiseFilter(6, $data)->count();
    $order['return_order'] = (new self)->statusWiseFilter(7, $data)->count();
    $order['packaging_order'] = (new self)->statusWiseFilter(8, $data)->count();
    $order['wfp_order'] = (new self)->statusWiseFilter(9, $data)->count();
    $order['partial_delivered'] = (new self)->statusWiseFilter(10, $data)->count();
    $order['return_pending'] = (new self)->statusWiseFilter(11, $data)->count();
    $order['incomplete_order'] = (new self)->statusWiseFilter(12, $data)->count();
    return $order;
  }


  public  function statusWiseFilter($status, $data)
  {

    $query = $this->allFilters($data);

    if ($data->query('status')) {
      $query->where('status', $status);
    }

    return $query;
  }

  public function allFilters($data)
  {
    $query =  Order::with(['customer:id,name,address,status,total_order', 'createAdmin:id,name', 'updateAdmin:id,name,image', 'approvedBy:id,name', 'courier:id,name,image', 'orderItem.product:id,name,thumbnail_img,product_code', 'orderItem.OrderItemVariant', 'source:id,name', 'reseller:id,name,phone']);
    /* ========= search filter ========= */

    /* ========= type filter ========= */
    if ($data->query('type')) {
      if ($data->query('type') != 'all') {
        $query->where('order_type', $data->type);
      }
    }
    $is_super_admin = session()->get('admin')['is_super_admin'] == 1;
    if (!$is_super_admin) {
      $query->where('assigned_admin_id', session()->get('admin')['id']);
    }

    /* ========= start date filter ========= */
    if (!empty($data->query('start_date')) && empty($data->query('end_date'))) {
      $query->whereDate('created_at', '=', $data->start_date);
    }
    /* ========= start date & end date filter ========= */
    if (!empty($data->query('start_date')) && !empty($data->query('end_date'))) {
      $query->whereDate('created_at', '>=', $data->start_date)
        ->whereDate('created_at', '<=', $data->end_date);
    }
    /* ========= courier filter ========= */
    if (!empty($data->query('courier_id'))) {
      $query->where('courier_id', $data->courier_id);
    }
    /* ========= city filter ========= */
    if (!empty($data->query('city_id'))) {
      $query->where('city_id', $data->city_id);
    }
    /* ========= source filter ========= */
    if (!empty($data->query('order_source'))) {
      $query->where('order_source', $data->order_source);
    }
    if (!empty($data->query('approved_admin_id'))) {
      $query->where('approved_admin_id', $data->approved_admin_id);
    }
    if (!empty($data->query('assigned_admin_id'))) {
      $query->where('assigned_admin_id', $data->assigned_admin_id);
    }
    return $query;
  }
  public static function adminOrderAnalysis()
  {
    $admin_id = session()->get('admin')['id'];
    $admin_order = array();
    $admin_order['session_admin'] = Order::where('create_admin_id', $admin_id)
      ->select(DB::raw('DATE(created_at) as created_at'), DB::raw('count(*) as total'))
      ->groupBy(DB::raw('DATE(created_at)'))
      ->orderBy('created_at', 'DESC')
      ->get();

    $admin_order['today'] = Order::whereNotNull('create_admin_id')
      ->where('created_at', '>=', Carbon::today()->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->select('create_admin_id', DB::raw('count(*) as total'), DB::raw('SUM(total) as total_amount'))
      ->groupBy('create_admin_id')
      ->orderBy('total', 'DESC')
      ->with('createAdmin')
      ->get();

    $admin_order['yesterday'] = Order::whereNotNull('create_admin_id')
      ->where('created_at', '>=', Carbon::yesterday()->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->select('create_admin_id', DB::raw('count(*) as total'), DB::raw('SUM(total) as total_amount'))
      ->groupBy('create_admin_id')
      ->orderBy('total', 'DESC')
      ->with('createAdmin')
      ->get();

    $admin_order['this_week'] = Order::whereNotNull('create_admin_id')
      ->where('created_at', '>=', Carbon::today()->subDays('7')->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->select('create_admin_id', DB::raw('count(*) as total'), DB::raw('SUM(total) as total_amount'))
      ->groupBy('create_admin_id')
      ->orderBy('total', 'DESC')
      ->with('createAdmin')
      ->get();

    $admin_order['this_month'] = Order::whereNotNull('create_admin_id')
      ->where('created_at', '>=', Carbon::today()->subDays('30')->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->select('create_admin_id', DB::raw('count(*) as total'), DB::raw('SUM(total) as total_amount'))
      ->groupBy('create_admin_id')
      ->orderBy('total', 'DESC')
      ->with('createAdmin')
      ->get();


    return $admin_order;
  }






  public static function profite()
  {
    $profite = array();
    $profite['today'] = 0;
    $today_order_items = OrderItem::where('created_at', '>=', Carbon::today()->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->get()
      ->map(function ($value) {
        return $value->price * $value->quantity;
      });
    return  $today_order_items;
  }



  public static function due()
  {
    $due = [];
    $due['office_sale_due'] = DB::table('customer_dues')->join('sales', 'customer_dues.sale_id', '=', 'sales.id')
      ->where('customer_dues.status', 0)
      ->where('customer_dues.amount', '>', 0)
      ->where('sales.company_id', null)
      ->sum('customer_dues.amount');

    $shipment_order = Order::where('status', 4)->get();
    $due['order_due'] = 0;
    $due['whole_sale_due'] = 00;
    foreach ($shipment_order as $item) {
      // return $item->order_type;
      if ($item->order_type != 3) {
        $due['order_due'] += ($item->total + $item->shipping_cost) - ($item->paid + $item->discount);
      }
      if ($item->order_type == 3) {
        $due['whole_sale_due'] += ($item->total + $item->shipping_cost) - ($item->paid + $item->discount);
      }
    }

    $a = SupplierPayment::all()->sum('amount');
    $b = Purchase::all()->sum('total');

    $product_supplier_id = Supplier::where('type', 1)->select('id')->pluck('id');
    $fabrice_supplier_id = Supplier::where('type', 2)->select('id')->pluck('id');

    $product_purchase_supplier_paid_amount = SupplierPayment::whereIn('supplier_id', $product_supplier_id)->sum('amount');
    $fabrics_purchase_supplier_paid_amount = SupplierPayment::whereIn('supplier_id', $fabrice_supplier_id)->sum('amount');
    $due['product_supplier'] =  Purchase::whereIn('supplier_id', $product_supplier_id)->sum('total') - $product_purchase_supplier_paid_amount;
    $due['fabrics_supplier'] = Purchase::whereIn('supplier_id', $fabrice_supplier_id)->sum('total') - $fabrics_purchase_supplier_paid_amount;
    return $due;
  }



  public static function analysis()
  {

    $analysis = [];
    $analysis['order'] = self::OrderAnalysis();
    $analysis['wholesale'] = self::WholeSaleAnalysis();
    $analysis['officeSale'] = self::SaleAnalysis();
    return $analysis;
  }

  public static function OrderAnalysis()
  {

    //get today order
    $today_orders = Order::where('created_at', '>=', Carbon::today()->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', '!=', 3)
      ->get();

    //get yesterday order
    $yesterday_orders = Order::where('created_at', '>=', Carbon::yesterday()->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', '!=', 3)
      ->get();
    //get this week order
    $this_week_orders = Order::where('created_at', '>=', Carbon::today()->subDays('7')->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', '!=', 3)
      ->get();

    //get this month order
    $this_month_orders = Order::where('created_at', '>=', Carbon::today()->subDays('30')->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', '!=', 3)
      ->get();


    $analysis = [];
    $product_quantity_today = 0;
    $product_quantity_yesterday = 0;
    $product_quantity_week = 0;
    $product_quantity_month = 0;

    $analysis['today'] = [];
    $analysis['yesterday'] = [];
    $analysis['this_week'] = [];
    $analysis['this_month'] = [];


    //today order summary
    foreach ($today_orders as $today_order) {
      $product_quantity_today += OrderItem::where('order_id', $today_order->id)->sum('quantity');
      $analysis['today'] = [
        'order_quantity' => $today_orders->count(),
        'product_quanity' => $product_quantity_today,
        'amount' => $today_orders->sum('total') - $today_orders->sum('discount')
      ];
    }


    //yesterday order summary
    foreach ($yesterday_orders as $yesterday_order) {
      $product_quantity_yesterday += OrderItem::where('order_id', $yesterday_order->id)->sum('quantity');
      $analysis['yesterday'] = [
        'order_quantity' => $yesterday_orders->count(),
        'product_quanity' => $product_quantity_yesterday,
        'amount' => $yesterday_orders->sum('total') - $yesterday_orders->sum('discount')
      ];
    }

    //this week  order summary
    foreach ($this_week_orders as $this_week_order) {
      $product_quantity_week += OrderItem::where('order_id', $this_week_order->id)->sum('quantity');
      $analysis['this_week'] = [
        'order_quantity' => $this_week_orders->count(),
        'product_quanity' => $product_quantity_week,
        'amount' => $this_week_orders->sum('total') - $this_week_orders->sum('discount')
      ];
    }

    //this month  order summary
    foreach ($this_month_orders as $this_month_order) {
      $product_quantity_month += OrderItem::where('order_id', $this_month_order->id)->sum('quantity');
      $analysis['this_month'] = [
        'order_quantity' => $this_month_orders->count(),
        'product_quanity' => $product_quantity_month,
        'amount' => $this_month_orders->sum('total') - $this_month_orders->sum('discount')
      ];
    }

    return $analysis;
  }


  public static function WholeSaleAnalysis()
  {

    //get today _whole_sales
    $today_whole_sales = Order::where('created_at', '>=', Carbon::today()->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', 3)
      ->get();

    //get yesterday _whole_sales
    $yesterday_whole_sales = Order::where('created_at', '>=', Carbon::yesterday()->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', 3)
      ->get();

    //get this week _whole_sales
    $this_week_whole_sales = Order::where('created_at', '>=', Carbon::yesterday()->subDays('7')->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', 3)
      ->get();

    //get this month _whole_sales
    $this_month_whole_sales = Order::where('created_at', '>=', Carbon::yesterday()->subDays('30')->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->where('status', '!=', 6)
      ->where('order_type', 3)
      ->get();
    //return $this_month_whole_sales->count();

    $analysis = [];
    $product_quantity_today = 0;
    $product_quantity_yesterday = 0;
    $product_quantity_week = 0;
    $product_quantity_month = 0;
    $analysis['today'] = [];
    $analysis['yesterday'] = [];
    $analysis['this_week'] = [];
    $analysis['this_month'] = [];


    //today _whole_sales summary
    foreach ($today_whole_sales as $today_whole_sale) {
      $product_quantity_today += OrderItem::where('order_id', $today_whole_sale->id)->sum('quantity');
      $analysis['today'] = [
        'order_quantity' => $today_whole_sales->count(),
        'product_quanity' => $product_quantity_today,
        'amount' => $today_whole_sales->sum('total') - $today_whole_sales->sum('discount')
      ];
    }

    //yesterday whole sales summary
    foreach ($yesterday_whole_sales as $yesterday_whole_sale) {
      $product_quantity_yesterday += OrderItem::where('order_id', $yesterday_whole_sale->id)->sum('quantity');
      $analysis['yesterday'] = [
        'order_quantity' => $yesterday_whole_sales->count(),
        'product_quanity' => $product_quantity_yesterday,
        'amount' => $yesterday_whole_sales->sum('total') - $yesterday_whole_sales->sum('discount')
      ];
    }

    //this week  _whole_sales summary
    foreach ($this_week_whole_sales as $this_week_whole_sale) {
      $product_quantity_week += OrderItem::where('order_id', $this_week_whole_sale->id)->sum('quantity');
      $analysis['this_week'] = [
        'order_quantity' => $this_week_whole_sales->count(),
        'product_quanity' => $product_quantity_week,
        'amount' => $this_week_whole_sales->sum('total') - $this_week_whole_sales->sum('discount')
      ];
    }

    //this month  _whole_sales summary
    foreach ($this_month_whole_sales as $this_month_whole_sale) {
      $product_quantity_month += OrderItem::where('order_id', $this_month_whole_sale->id)->sum('quantity');
      $analysis['this_month'] = [
        'order_quantity' => $this_month_whole_sales->count(),
        'product_quanity' => $product_quantity_month,
        'amount' => $this_month_whole_sales->sum('total') - $this_month_whole_sales->sum('discount')
      ];
    }

    return $analysis;
  }

  public static function SaleAnalysis()
  {

    //get today sales
    $today_sales = Sale::where('created_at', '>=', Carbon::today()->startOfDay())
      ->where('created_at', '<=', Carbon::today()->endOfDay())
      ->get();

    //get yester sales
    $yesterday_sales = Sale::where('created_at', '>=', Carbon::yesterday()->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->get();

    //get this week _whole_sales
    $this_week_sales = Sale::where('created_at', '>=', Carbon::yesterday()->subDays('7')->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->get();

    //get this month order
    $this_monthe_sales = Sale::where('created_at', '>=', Carbon::yesterday()->subDays('30')->startOfDay())
      ->where('created_at', '<=', Carbon::yesterday()->endOfDay())
      ->get();


    $analysis = [];
    $product_quantity_today = 0;
    $product_quantity_yesterday = 0;
    $product_quantity_week = 0;
    $product_quantity_month = 0;
    $analysis['today'] = [];
    $analysis['yesterday'] = [];
    $analysis['this_week'] = [];
    $analysis['this_month'] = [];


    //today order summary
    foreach ($today_sales as $today_sale) {
      $product_quantity_today += SaleItem::where('sale_id', $today_sale->id)->sum('qty');
      $analysis['today'] = [
        'order_quantity' => $today_sales->count(),
        'product_quanity' => $product_quantity_today,
        'amount' => $today_sales->sum('total') - $today_sales->sum('discount')
      ];
    }
    //yesterday order summary
    foreach ($yesterday_sales as $yesterday_sale) {
      $product_quantity_yesterday += SaleItem::where('sale_id', $yesterday_sale->id)->sum('qty');
      $analysis['yesterday'] = [
        'order_quantity' => $yesterday_sales->count(),
        'product_quanity' => $product_quantity_yesterday,
        'amount' => $yesterday_sales->sum('total') - $yesterday_sales->sum('discount'),
      ];
    }
    //this week  order summary
    foreach ($this_week_sales as $this_week_sale) {
      $product_quantity_week += SaleItem::where('sale_id', $this_week_sale->id)->sum('qty');
      $analysis['this_week'] = [
        'order_quantity' => $this_week_sales->count(),
        'product_quanity' => $product_quantity_week,
        'amount' => $this_week_sales->sum('total') - $this_week_sales->sum('discount')
      ];
    }

    //this month  order summary
    foreach ($this_monthe_sales as $this_monthe_sale) {
      $product_quantity_month += SaleItem::where('sale_id', $this_monthe_sale->id)->sum('qty');
      $analysis['this_month'] = [
        'order_quantity' => $this_monthe_sales->count(),
        'product_quanity' => $product_quantity_month,
        'amount' => $this_monthe_sales->sum('total') - $this_monthe_sales->sum('discount')
      ];
    }

    return $analysis;
  }


  public static function orderPressingReport($admin_id, $start_date, $end_date)
  {
    $approvedOrders = Order::where('approved_admin_id', $admin_id)
      ->whereBetween('approved_date', [$start_date, $end_date]);

    $data = [
      'created' => Order::where('create_admin_id', $admin_id)
        ->whereBetween('created_at', [$start_date, $end_date])
        ->count(),

      'pending' => Order::where('pending_admin_id', $admin_id)
        ->whereBetween('pending_date', [$start_date, $end_date])
        ->count(),

      'approved' => $approvedOrders->count(),

      'shipment' => (clone $approvedOrders)->where('status', 4)->count(),

      'delivered' => (clone $approvedOrders)->where('status', 5)->count(),

      'returned' => (clone $approvedOrders)->whereIn('status', [7, 11])->count(),

      'canceled' => Order::where('cancel_admin_id', $admin_id)
        ->where('status', 6)
        ->whereBetween('cancel_date', [$start_date, $end_date])
        ->count(),
    ];

    return $data;
  }
}
