<?php

namespace App\Services;

use App\Models\Product;
use App\Models\OrderItem;
use Illuminate\Support\Carbon;
use App\Models\CustomerCashbook;
use Illuminate\Support\Facades\DB;
use App\Models\CustomerPaymentTransaction;

class HomeService
{
  public function getSliders()
  {
    return  DB::table('sliders')->select('id', 'image', 'status', 'url')
      ->where('status', 1)
      ->inRandomOrder()
      ->latest()
      ->limit(5)->get();
  }

  public function getTopSectionCategories()
  {
    return DB::table('categories')->select('id', 'name', 'slug', 'icon_image')
      ->where('status', 1)
      ->orderBy('position', 'asc')
      ->inRandomOrder()
      ->latest()
      ->limit(10)
      ->get();
  }
  
  public function getTopSectionBrand()
  {
    return DB::table('brands')->select('id', 'name', 'slug', 'image')
      ->where('status', 1)
      ->orderBy('position', 'asc')
      ->inRandomOrder()
      ->latest()
      ->limit(10)
      ->get();
  }

  public function getTopSellingProducts()
  {

    $top_selling_product_id = OrderItem::where('created_at', '>=', Carbon::today()->subDays('7'))
      ->select('product_id', DB::raw('count(*) as total'))
      ->groupBy('product_id')
      ->orderBy('total', 'desc')
      ->take(12)
      ->pluck('product_id');
    return Product::WhereIn('id', $top_selling_product_id)
      ->with('author:id,name', 'publisher:id,name')
      ->select('id', 'name', 'price', 'sale_price', 'reselling_price', 'slug', 'discount', 'thumbnail_img', 'stock', 'is_free_delivery', 'review_ratings', 'avg_review', 'author_id', 'publisher_id')
      ->where('status', 1)
      ->take(10)
      ->get();
  }
  public function getNewProducts()
  {

    return Product::where('status', 1)
      ->with('author:id,name', 'publisher:id,name')
      ->withAvg('productReviews', 'rating_stars')
      ->select('id', 'name', 'price', 'sale_price', 'reselling_price', 'slug', 'discount', 'thumbnail_img', 'stock', 'is_free_delivery', 'review_ratings', 'avg_review', 'author_id', 'publisher_id')
      ->orderBy('id', 'desc')
      ->take(10)
      ->get();
  }

  public function getBanners($placements, $limit = null)
  {
    $query = DB::table('banners')
      ->select('id', 'url', 'banner', 'placement', 'status')
      ->whereIn('placement', (array) $placements)
      ->where('status', 1)
      ->inRandomOrder();

    return $limit === 1 ? $query->first() : $query->limit($limit)->get();
  }

  public function getProductReviews()
  {

    return DB::table('product_reviews')->select('image')
      ->where('status', 1)
      ->latest()
      ->limit(10)
      ->get();
  }

  public function getAuthors()
  {

    return DB::table('authors')->select('id', 'name', 'slug', 'image', 'description')
      ->where('status', 1)
      ->orderBy('id', 'asc')
      ->inRandomOrder()
      ->latest()
      ->limit(8)
      ->get();
  }

  public function getPublishers()
  {

    return DB::table('publishers')->select('id', 'name', 'slug', 'image')
      ->where('status', 1)
      ->orderBy('id', 'asc')
      ->inRandomOrder()
      ->latest()
      ->limit(8)
      ->get();
  }

  public function getTopAuthors()
  {

    return OrderItem::join('products', 'order_items.product_id', '=', 'products.id')
      ->join('authors', 'products.author_id', '=', 'authors.id')
      ->select(
        'authors.id as author_id',
        'authors.name as author_name',
        'authors.image as author_image',
        'authors.description as author_description',
        'products.id as book_id',
        'products.name as book_name',
        'products.thumbnail_img as thumbnail_img',
        'products.slug as slug',
        DB::raw('SUM(order_items.quantity) as total_orders')
      )
      ->groupBy('authors.id', 'authors.name', 'authors.image', 'authors.description', 'products.id', 'products.name', 'products.thumbnail_img', 'products.slug')
      ->orderByDesc('total_orders')
      ->get()
      ->groupBy('author_id')
      ->sortByDesc(fn($books) => $books->sum('total_orders'))
      ->first();
  }

  
  public function getProducts($category_column_name, $category_id, $items, $sortBy)
  {
      $query = Product::select('id', 'name', 'product_code', 'price', 'sale_price', 'slug', 'stock', 'discount', 'thumbnail_img', 'is_free_delivery', 'review_ratings', 'avg_review', 'author_id', 'publisher_id', 'created_at', 'product_position')
          ->with([
              'productVariant:id,product_id,variant_id,attribute_id,price', 
              'productVariant.variant:id,name,status', 
              'productVariant.attribute:id,name', 
              'author:id,name', 
              'publisher:id,name'
          ])
          ->where($category_column_name, $category_id)
          ->where('status', 1)
          ->where('show_homepage', 1);

      if ($sortBy==1) {
          $query->orderBy('product_position', 'asc');
      } else {
          $query->orderBy('id', 'desc');
      }
  
      return $query->take($items)->get();
  }
  public function saveCustomerWithdrawRequest($customer, $request, $type,$amount)
  {
      $withdrawal = new CustomerCashbook();
      $withdrawal->payment_method = $request->payment_method;
      $withdrawal->note = $request->note;
      $withdrawal->is_income = 0;
      $withdrawal->type = $type;
      $withdrawal->customer_id = $customer->id;
      $withdrawal->amount = $amount;
      $withdrawal->save();
  }
  public function saveCustomerPaymentTransaction($customer, $request, $type,$amount)
  {
      $transaction = new CustomerPaymentTransaction();
      $transaction->amount = $amount;
      $transaction->customer_id = $customer->id;
      $transaction->note = $request->note;
      $transaction->type = $type;
      $transaction->payment_method = $request->payment_method;
      $transaction->transaction_id = 'TNX-' . rand(1111, 9999) . $transaction->id;
      $transaction->save();
  }
}